<?php
require __DIR__ . '/../db.php';
header('Content-Type: application/json');

$action = $_GET['action'] ?? '';
if (!in_array($action, ['csrf','settings_get','admin_me','features_list','categories_list','duplicate_suggest_lev'])) {
  require_csrf();
}

function read_json(){
  $in = file_get_contents('php://input');
  $data = json_decode($in,true);
  return is_array($data) ? $data : [];
}
function out($data,$code=200){
  http_response_code($code);
  echo json_encode($data);
  exit;
}

switch ($action) {
  /* --- basic --- */
  case 'csrf':
    out(['csrf'=>$_SESSION['csrf']]);

  case 'settings_get':
    out(['allowed_domain'=>get_setting($db,'allowed_domain','')]);

  case 'admin_me':
    out(['user'=>current_user($db)]);

  /* --- admin login (password) --- */
  case 'admin_login':
    $d = read_json();

    // First time: set default password for FeatureAdmin if missing
    if (($d['identifier'] ?? '') === 'FeatureAdmin') {
      $row = $db->queryOne("SELECT id,password_hash FROM users WHERE name='FeatureAdmin' AND role='admin' LIMIT 1", []);
      if ($row && !$row['password_hash']) {
        $hash = password_hash('!1Feature3#', PASSWORD_BCRYPT);
        $db->exec("UPDATE users SET password_hash=? WHERE id=?", [$hash,$row['id']]);
      }
    }

    list($ok,$msg) = admin_login($db, $d['identifier'] ?? '', $d['password'] ?? '');
    if (!$ok) out(['error'=>$msg],401);
    out(['ok'=>true]);

  /* --- settings --- */
  case 'settings_set_domain':
    require_admin($db);
    $d = read_json();
    $dom = strtolower(trim($d['allowed_domain'] ?? ''));
    if (!$dom) out(['error'=>'invalid_domain'],422);
    set_setting($db,'allowed_domain',$dom);
    out(['ok'=>true]);

  /* --- categories --- */
  case 'categories_list':
    $rows = $db->queryAll("SELECT id,name,sort_order FROM categories ORDER BY sort_order,name");
    out($rows);

  case 'categories_create':
    require_admin($db);
    $d = read_json();
    $name = trim($d['name'] ?? '');
    $sort = (int)($d['sort_order'] ?? 100);
    if ($name==='') out(['error'=>'name_required'],422);
    $db->exec("INSERT INTO categories (name,sort_order) VALUES (?,?)", [$name,$sort]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'categories_delete':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $db->exec("DELETE FROM categories WHERE id=?", [$id]);
    out(['ok'=>true]);

  /* --- magic link request --- */
  case 'request_magic_link':
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) out(['error'=>'invalid_email'],422);

    $allowed = get_setting($db,'allowed_domain','');
    $domain = substr(strrchr($email,'@'),1);
    if ($allowed && strtolower($domain) !== strtolower($allowed)) {
      out(['error'=>'domain_not_allowed'],403);
    }

    $uid = get_or_create_user_by_email($db,$email);
    $token = bin2hex(random_bytes(32));
    $db->exec(
      "UPDATE users SET magic_token=?, magic_token_expires = DATE_ADD(NOW(), INTERVAL 15 MINUTE) WHERE id=?",
      [$token,$uid]
    );

    // send email (adjust From: line as needed)
    $link = 'https://fieldsupportcenter.com/featureboard/magic.php?token='.$token;
    $subject = 'Your FeatureBoard sign-in link';
    $body = "Click this link to sign in:\n\n".$link."\n\nThis link expires in 15 minutes.";
    $headers = "From: no-reply@fieldsupportcenter.com\r\n";
    @mail($email, $subject, $body, $headers);

    out(['ok'=>true]);

  /* --- user/admin management --- */
  case 'users_create_admin':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $name  = trim($d['name'] ?? $email);
    $pass  = $d['password'] ?? '';
    if (!filter_var($email,FILTER_VALIDATE_EMAIL) || strlen($pass)<6) out(['error'=>'invalid'],422);
    $row = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if ($row) out(['error'=>'exists'],409);
    $hash = password_hash($pass, PASSWORD_BCRYPT);
    $db->exec("INSERT INTO users (email,name,role,password_hash) VALUES (?,?,?,?)",[$email,$name,'admin',$hash]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'users_set_role':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $role  = $d['role'] ?? 'user';
    if (!in_array($role,['user','admin'])) out(['error'=>'invalid_role'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $db->exec("UPDATE users SET role=? WHERE id=?",[$role,$u['id']]);
    out(['ok'=>true]);

  case 'users_set_password':
    require_admin($db);
    $d = read_json();
    $email = normalize_email($d['email'] ?? '');
    $pass = $d['password'] ?? '';
    if (!$email || strlen($pass)<6) out(['error'=>'invalid'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $hash = password_hash($pass,PASSWORD_BCRYPT);
    $db->exec("UPDATE users SET password_hash=? WHERE id=?",[$hash,$u['id']]);
    out(['ok'=>true]);

  case 'admin_users_list':
    require_admin($db);
    $rows = $db->queryAll("
      SELECT
        u.id,u.email,u.name,u.role,u.last_seen_at,
        SUM(CASE f.status_id WHEN 1 THEN 1 ELSE 0 END) AS cnt_backlog,
        SUM(CASE f.status_id WHEN 2 THEN 1 ELSE 0 END) AS cnt_planned,
        SUM(CASE f.status_id WHEN 3 THEN 1 ELSE 0 END) AS cnt_progress,
        SUM(CASE f.status_id WHEN 4 THEN 1 ELSE 0 END) AS cnt_released,
        SUM(CASE f.status_id WHEN 5 THEN 1 ELSE 0 END) AS cnt_rejected
      FROM users u
      LEFT JOIN features f ON f.author_id = u.id
      GROUP BY u.id
      ORDER BY u.email
    ");
    out($rows);

  case 'admin_user_detail':
    require_admin($db);
    $id = (int)($_GET['id'] ?? 0);
    $user = $db->queryOne("SELECT id,email,name,role,last_seen_at FROM users WHERE id=?",[$id]);
    if (!$user) out(['error'=>'not_found'],404);
    $features = $db->queryAll("
      SELECT f.id,f.title,f.status_id,f.created_at,f.updated_at
      FROM features f
      WHERE f.author_id=?
      ORDER BY f.created_at DESC
    ",[$id]);
    out(['user'=>$user,'features'=>$features]);

  /* --- features (public list & admin list) --- */
  case 'features_list':
    $sort = $_GET['sort'] ?? 'top';
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : null;
    $status = isset($_GET['status']) ? (int)$_GET['status'] : null;

    $where=[]; $params=[];
    if ($category_id){ $where[]="f.category_id=?"; $params[]=$category_id; }
    if ($status){ $where[]="f.status_id=?"; $params[]=$status; }
    $w = $where ? ("WHERE ".implode(" AND ",$where)) : "";

    if ($sort==='new')      $order="ORDER BY f.created_at DESC";
    elseif ($sort==='trending') $order="ORDER BY recent_votes DESC,total_votes DESC,f.created_at DESC";
    else                    $order="ORDER BY total_votes DESC,f.created_at DESC";

    $sql = "
      SELECT f.*,COALESCE(tv.total_votes,0) AS total_votes,
             COALESCE(rv.recent_votes,0) AS recent_votes,
             c.name AS category_name
      FROM features f
      LEFT JOIN (SELECT feature_id,COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
        ON tv.feature_id=f.id
      LEFT JOIN (
        SELECT feature_id,COUNT(*) recent_votes
        FROM feature_votes
        WHERE created_at >= (NOW() - INTERVAL 30 DAY)
        GROUP BY feature_id
      ) rv ON rv.feature_id=f.id
      LEFT JOIN categories c ON c.id=f.category_id
      $w
      $order
      LIMIT 200";
    $rows = $db->queryAll($sql,$params);
    out($rows);

  case 'admin_features_list':
    require_admin($db);
    $status = isset($_GET['status']) ? (int)$_GET['status'] : null;
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : null;
    $q = trim($_GET['q'] ?? '');

    $where=[]; $params=[];
    if ($status){ $where[]="f.status_id=?"; $params[]=$status; }
    if ($category_id){ $where[]="f.category_id=?"; $params[]=$category_id; }
    if ($q!==''){ $where[]="f.title LIKE ?"; $params[]='%'.$q.'%'; }
    $w = $where ? ("WHERE ".implode(" AND ",$where)) : "";

    $rows = $db->queryAll("
      SELECT f.id,f.title,f.status_id,f.category_id,f.duplicate_of,
             u.email AS author_email,
             COALESCE(tv.total_votes,0) AS total_votes,
             f.created_at,f.updated_at
      FROM features f
      LEFT JOIN users u ON u.id=f.author_id
      LEFT JOIN (SELECT feature_id,COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
        ON tv.feature_id=f.id
      $w
      ORDER BY f.created_at DESC
      LIMIT 500
    ",$params);
    out($rows);

  case 'admin_feature_update':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $fields=[]; $params=[];
    if (isset($d['title'])) { $fields[]="title=?"; $params[] = trim($d['title']); }
    if (isset($d['description'])) { $fields[]="description=?"; $params[] = trim($d['description']); }
    if (isset($d['status_id'])) { $fields[]="status_id=?"; $params[] = (int)$d['status_id']; }
    if (isset($d['category_id'])) { $fields[]="category_id=?"; $params[] = $d['category_id'] ? (int)$d['category_id'] : null; }
    if (!$fields) out(['error'=>'nothing_to_update'],422);
    $params[]=$id;
    $db->exec("UPDATE features SET ".implode(",",$fields)." WHERE id=?",$params);
    out(['ok'=>true]);

  case 'admin_feature_delete':
    require_admin($db);
    $d = read_json();
    $id = (int)($d['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $db->exec("DELETE FROM features WHERE id=?",[$id]);
    out(['ok'=>true]);

  case 'features_mark_duplicate':
    require_admin($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    $dup = (int)($d['duplicate_of_id'] ?? 0);
    if (!$fid || !$dup || $fid===$dup) out(['error'=>'invalid_ids'],422);
    $has = $db->queryOne("SELECT id FROM features WHERE id=?",[$dup]);
    if (!$has) out(['error'=>'dup_not_found'],404);
    $db->exec("UPDATE features SET duplicate_of=? WHERE id=?",[$dup,$fid]);
    out(['ok'=>true]);

  /* --- public create / vote / comment --- */
  case 'features_create':
    require_user($db);
    $d = read_json();
    $title = trim($d['title'] ?? '');
    $desc  = trim($d['description'] ?? '');
    $catId = isset($d['category_id']) ? (int)$d['category_id'] : null;
    if ($title==='') out(['error'=>'title_required'],422);
    $u = current_user($db);
    touch_user($db, $u['id']);
    $db->exec(
      "INSERT INTO features (title,description,author_id,category_id) VALUES (?,?,?,?)",
      [$title,$desc,$u['id'],$catId ?: null]
    );
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'vote_toggle':
    require_user($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    if (!$fid) out(['error'=>'feature_id_required'],422);
    $u = current_user($db);
    touch_user($db,$u['id']);
    $uid = $u['id'];
    $exists = $db->queryOne(
      "SELECT 1 FROM feature_votes WHERE feature_id=? AND user_id=?",
      [$fid,$uid]
    );
    if ($exists){
      $db->exec("DELETE FROM feature_votes WHERE feature_id=? AND user_id=?",[$fid,$uid]);
      out(['ok'=>true,'action'=>'unvoted']);
    } else {
      $db->exec("INSERT INTO feature_votes (feature_id,user_id) VALUES (?,?)",[$fid,$uid]);
      out(['ok'=>true,'action'=>'voted']);
    }

  case 'comments_create':
    require_user($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    $body= trim($d['body'] ?? '');
    if (!$fid || $body==='') out(['error'=>'invalid'],422);
    $u = current_user($db);
    touch_user($db,$u['id']);
    $db->exec("INSERT INTO feature_comments (feature_id,user_id,body) VALUES (?,?,?)",
      [$fid,$u['id'],$body]);
    out(['ok'=>true]);

  /* --- Levenshtein duplicate suggestions --- */
  case 'duplicate_suggest_lev':
    $q = trim($_GET['q'] ?? '');
    if ($q==='') out([]);
    $all = $db->queryAll("SELECT id,title FROM features ORDER BY id DESC LIMIT 500");
    $res = [];
    foreach ($all as $r) {
      $dist = levenshtein(mb_strtolower($q,'UTF-8'), mb_strtolower($r['title'],'UTF-8'));
      $res[] = ['id'=>$r['id'], 'title'=>$r['title'], 'distance'=>$dist];
    }
    usort($res, function($a,$b){ return $a['distance'] <=> $b['distance']; });
    out(array_slice($res,0,10));

  default:
    out(['error'=>'unknown_action'],400);
}
