<?php
require __DIR__ . '/../db.php';
$u = current_user($db);
$isAdmin = $u && $u['role']==='admin';
?>
<!doctype html>
<html lang="en">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width,initial-scale=1" />
<title>Admin – FeatureBoard</title>
<style>
/* Brand + palette */
:root {
  --color-1: #ffffff;
  --color-2: #afbac5;
  --color-3: #7f9aad;
  --color-4: #5c92c1;
  --color-5: #7890a1;

  --primary: var(--color-4);
  --primary-dark: #325a8b;
  --bg-soft: #f3f5f8;
  --card-bg: var(--color-1);
  --border-soft: #dde3ea;
  --text-main: #2f343b;
  --text-muted: #6c7175;
}

body{
  font-family:system-ui,-apple-system,Segoe UI,Roboto,Ubuntu;
  margin:0;
  background:var(--bg-soft);
  color:var(--text-main);
}
.wrap{
  max-width:1100px;
  margin:24px auto;
  padding:0 16px;
}
.card{
  background:var(--card-bg);
  border:1px solid var(--border-soft);
  border-radius:12px;
  box-shadow:0 2px 6px rgba(0,0,0,.04);
  padding:16px;
  margin-bottom:16px;
}
.row{
  display:flex;
  gap:12px;
  align-items:center;
  flex-wrap:wrap;
}

/* Primary button */
.btn{
  border:1px solid var(--primary);
  background:var(--primary);
  color:#fff;
  border-radius:10px;
  padding:6px 12px;
  cursor:pointer;
  font-size:14px;
  transition:background .15s,border-color .15s,transform .05s;
}
.btn:hover{
  background:var(--primary-dark);
  border-color:var(--primary-dark);
  transform:translateY(-1px);
}

/* Neutral button */
.btn-secondary{
  border:1px solid var(--border-soft);
  background:var(--card-bg);
  color:var(--text-main);
}
.btn-secondary:hover{
  background:#f3f5f8;
}

input,select,textarea{
  border:1px solid var(--border-soft);
  border-radius:10px;
  padding:6px 8px;
  font-size:14px;
}
.muted{
  color:var(--text-muted);
}

/* Tabs for Settings / Features / Users / Roadmap */
.tabs{
  display:flex;
  gap:8px;
  margin-bottom:12px;
}
.tabbtn{
  padding:6px 10px;
  border-radius:999px;
  border:1px solid var(--border-soft);
  background:#f9fafb;
  cursor:pointer;
  font-size:14px;
  color:var(--text-main);
}
.tabbtn.active{
  background:var(--primary);
  color:#f9fafb;
  border-color:var(--primary);
}
.tab{display:none}
.tab.active{display:block}

/* Tables */
table{
  border-collapse:collapse;
  width:100%;
  font-size:13px;
}
th,td{
  border-bottom:1px solid var(--border-soft);
  padding:6px 4px;
  text-align:left;
}
th{font-weight:600}

/* Status badges */
.badge{
  font-size:11px;
  border-radius:999px;
  border:1px solid #e5e7eb;
  padding:2px 6px;
}
.status-badge-1{background:#eff6ff;border-color:#bfdbfe}
.status-badge-2{background:#ecfeff;border-color:#bae6fd}
.status-badge-3{background:#fef9c3;border-color:#facc15}
.status-badge-4{background:#dcfce7;border-color:#86efac}
.status-badge-5{background:#fee2e2;border-color:#fecaca}

.mb8{margin-bottom:8px}

/* Inline edit panel for features */
.edit-card{
  background:#f9fafb;
  border-radius:10px;
  border:1px solid var(--border-soft);
  padding:10px;
  font-size:13px;
}
.edit-row-grid{
  display:grid;
  grid-template-columns:minmax(0,2fr) minmax(0,1fr);
  gap:12px;
}
@media (max-width:700px){
  .edit-row-grid{grid-template-columns:1fr;}
}

.site-header{
  background:var(--card-bg);
  border-bottom:1px solid var(--border-soft);
  box-shadow:0 2px 4px rgba(0,0,0,.03);
}
.header-inner{
  max-width:1100px;
  margin:0 auto;
  padding:10px 16px;
  display:flex;
  align-items:center;
  gap:16px;
}
.header-inner img{
  height:40px;
}
.header-title{
  font-size:20px;
  font-weight:600;
  color:var(--primary);
}

/* Roadmap board (admin) */
.roadmap-board{
  display:flex;
  gap:12px;
  overflow-x:auto;
}
.roadmap-column{
  flex:1;
  min-width:220px;
  background:#f9fafb;
  border-radius:10px;
  border:1px solid var(--border-soft);
  padding:8px;
  display:flex;
  flex-direction:column;
  max-height:500px;
}
.roadmap-column-header{
  font-weight:600;
  font-size:14px;
  margin-bottom:6px;
}
.roadmap-column-body{
  flex:1;
  overflow-y:auto;
}
.roadmap-card{
  background:#fff;
  border-radius:8px;
  border:1px solid var(--border-soft);
  padding:6px;
  margin-bottom:6px;
  cursor:grab;
  font-size:12px;
}
.roadmap-card-title{
  font-weight:600;
  margin-bottom:2px;
}
.roadmap-column.drag-over{
  border-color:var(--primary);
  box-shadow:0 0 0 2px rgba(92,146,193,.25);
}
</style>

</head>
<body>
<header class="site-header">
  <div class="header-inner">
    <img src="../images/ailogo.png"
         alt="AGRIntelligence logo"
         onerror="this.style.display='none'">
  </div>
</header>

<div class="wrap">
  <h1>Admin</h1>

  <!-- Login card -->
  <div id="login-card" class="card" style="<?php echo $isAdmin ? 'display:none;' : '' ?>">
    <h2>Admin login</h2>
    <p class="muted">Sign in as <strong>FeatureAdmin</strong> with the default password on first run, then change or create your own admin accounts.</p>
    <div class="row mb8">
      <input id="adm-ident" placeholder="FeatureAdmin or admin email" />
      <input id="adm-pass" placeholder="Password" type="password" />
      <button class="btn" id="btn-adm-login">Login</button>
    </div>
    <div id="adm-msg" class="muted"></div>
  </div>

  <!-- Main admin UI -->
  <div id="admin-ui" style="<?php echo $isAdmin ? '' : 'display:none;' ?>">

    <div class="card">
      <div class="row mb8" style="justify-content:space-between;">
        <div class="muted">Logged in as <strong><?php echo $isAdmin ? htmlspecialchars($u['email']) : '' ?></strong></div>
        <button class="btn" id="btn-logout">Log out</button>
      </div>
      <div class="tabs">
        <button class="tabbtn active" data-tab="settings">Settings</button>
        <button class="tabbtn" data-tab="features">Features</button>
        <button class="tabbtn" data-tab="users">Users</button>
        <button class="tabbtn" data-tab="roadmap">Roadmap</button>
      </div>
    </div>

    <!-- SETTINGS TAB -->
    <div class="card tab active" id="tab-settings">
      <h2>Settings</h2>

      <h3>Allowed email domain</h3>
      <div class="row mb8">
        <input id="set-domain" style="max-width:260px" />
        <button class="btn" id="btn-save-domain">Save</button>
        <span class="muted" id="set-domain-msg"></span>
      </div>
      
      <h3>Submission limit (per non-admin user)</h3>
      <div class="row mb8">
        <input id="set-limit" style="max-width:120px" placeholder="0 = unlimited" />
        <button class="btn btn-secondary" id="btn-save-limit">Save</button>
        <span class="muted" id="set-limit-msg"></span>
      </div>

      <h3>Vote limit (per non-admin user)</h3>
      <div class="row mb8">
        <input id="set-vote-limit" style="max-width:120px" placeholder="0 = unlimited" />
        <button class="btn btn-secondary" id="btn-save-vote-limit">Save</button>
        <span class="muted" id="set-vote-limit-msg"></span>
      </div>

      <h3>Categories</h3>
      <div class="row mb8">
        <input id="cat-name" placeholder="Category name" />
        <input id="cat-sort" placeholder="Sort" style="max-width:80px" />
        <button class="btn" id="btn-add-cat">Add</button>
      </div>
      <table id="cat-table">
        <thead><tr><th>Name</th><th>Sort</th><th></th></tr></thead>
        <tbody></tbody>
      </table>
    </div>

    <!-- FEATURES TAB -->
    <div class="card tab" id="tab-features">
      <div class="row mb8" style="justify-content:space-between;align-items:center;">
        <h2 style="margin:0;">Features</h2>
        <div class="row" style="gap:8px;align-items:center;">
          <button class="btn btn-secondary" id="btn-export-feats" type="button">
            Export CSV
          </button>

          <input type="file"
                 id="feat-import-file"
                 accept=".csv"
                 style="display:none;" />

          <button class="btn btn-secondary" id="btn-import-feats" type="button">
            Import CSV
          </button>

          <button class="btn btn-secondary" id="btn-toggle-new-feat" type="button">
            Add feature
          </button>
        </div>
      </div>

      <div class="row mb8">
        <select id="filt-status">
          <option value="">All statuses</option>
          <option value="1">Backlog</option>
          <option value="2">Planned</option>
          <option value="3">In Progress</option>
          <option value="4">Released</option>
          <option value="5">Rejected</option>
        </select>
        <select id="filt-cat"></select>
        <input id="filt-q" placeholder="Search title" style="min-width:200px" />
        <button class="btn" id="btn-filt">Apply</button>
      </div>

      <!-- Hidden admin-only "create feature" form -->
      <div class="mb8" id="new-feat-card" style="margin-bottom:16px;display:none;">
        <div class="edit-card">
          <div class="edit-row-grid">
            <div>
              <div class="mb8">
                <div class="muted">Title</div>
                <input id="new-feat-title" />
              </div>
              <div class="mb8">
                <div class="muted">Description</div>
                <textarea id="new-feat-desc" rows="3"></textarea>
              </div>
              <div class="mb8">
                <div class="muted">Author email</div>
                <input id="new-feat-author" placeholder="user@helenaagri.com" />
              </div>
            </div>
            <div>
              <div class="mb8">
                <div class="muted">Status</div>
                <select id="new-feat-status">
                  <option value="1">Backlog</option>
                  <option value="2">Planned</option>
                  <option value="3">In Progress</option>
                  <option value="4">Released</option>
                  <option value="5">Rejected</option>
                </select>
              </div>
              <div class="mb8">
                <div class="muted">Category</div>
                <select id="new-feat-cat">
                  <option value="">(none)</option>
                  <!-- options filled from JS -->
                </select>
              </div>
              <div class="mb8">
                <button class="btn" id="btn-new-feat-save" type="button">Create feature</button>
                <button class="btn btn-secondary" id="btn-new-feat-clear" type="button">Clear</button>
                <span class="muted" id="new-feat-msg"></span>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div class="muted" id="feat-import-msg" style="font-size:13px;margin-bottom:8px;"></div>

      <table id="feat-table">
        <thead>
          <tr>
            <th>ID</th><th>Title</th><th>Status</th><th>Category</th><th>Votes</th><th>Author</th><th>Dup Of</th><th>Actions</th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
    </div>

    <!-- USERS TAB -->
    <div class="card tab" id="tab-users">
      <h2>Users</h2>
      <h3>Create admin</h3>
      <div class="row mb8">
        <input id="u-email" placeholder="email@example.com" />
        <input id="u-name" placeholder="Name (optional)" />
        <input id="u-pass" placeholder="Password" type="password" />
        <button class="btn" id="btn-create-admin">Create admin</button>
      </div>
      <h3>All users</h3>
      <table id="user-table">
        <thead>
          <tr>
            <th>Email</th><th>Admin</th><th>Last seen</th>
            <th>B</th><th>P</th><th>IP</th><th>R</th><th>Rej</th><th></th>
          </tr>
        </thead>
        <tbody></tbody>
      </table>
      <div id="user-detail" style="margin-top:16px;"></div>
    </div>

    <!-- ROADMAP TAB -->
    <div class="card tab" id="tab-roadmap">
      <h2>Roadmap</h2>
      <p class="muted">Drag items between columns to change status. This only affects the internal status field.</p>
    
      <div class="roadmap-board" id="roadmap-board">
        <div class="roadmap-column" data-status="1">
          <div class="roadmap-column-header">Backlog</div>
          <div class="roadmap-column-body" id="roadmap-col-1" data-status="1"></div>
        </div>
        <div class="roadmap-column" data-status="2">
          <div class="roadmap-column-header">Planned</div>
          <div class="roadmap-column-body" id="roadmap-col-2" data-status="2"></div>
        </div>
        <div class="roadmap-column" data-status="3">
          <div class="roadmap-column-header">In Progress</div>
          <div class="roadmap-column-body" id="roadmap-col-3" data-status="3"></div>
        </div>
        <div class="roadmap-column" data-status="4">
          <div class="roadmap-column-header">Released</div>
          <div class="roadmap-column-body" id="roadmap-col-4" data-status="4"></div>
        </div>
        <div class="roadmap-column" data-status="5">
          <div class="roadmap-column-header">Rejected</div>
          <div class="roadmap-column-body" id="roadmap-col-5" data-status="5"></div>
        </div>
      </div>
    
      <p class="muted" style="margin-top:8px;">
        Public read-only roadmap: 
        <a class="btn btn-secondary" href="roadmap.php" target="_blank">Open in new tab</a>
      </p>
    </div>

  </div>
</div>

<script>
let CSRF = null;
async function getCsrf(){
  const r = await fetch('api.php?action=csrf');
  const j = await r.json();
  CSRF = j.csrf;
}
async function api(path,opts={}) {
  const o = Object.assign({headers:{}}, opts);
  if (!o.headers['Content-Type'] && o.body) {
    o.headers['Content-Type'] = 'application/json';
  }
  if (!o.headers['X-CSRF-Token']) {
    o.headers['X-CSRF-Token'] = CSRF;
  }
  
  // 🔄 Always bypass cache so fresh data is fetched
  if (!o.cache) {
    o.cache = 'no-store';
  }

  let r = await fetch('api.php' + path, o);
  let j = await r.json();

  if (j && j.error === 'csrf_failed') {
    try {
      await getCsrf();
      o.headers['X-CSRF-Token'] = CSRF;
      r = await fetch('api.php' + path, o);
      j = await r.json();
    } catch (e) {}
    if (j && j.error === 'csrf_failed') {
      alert(j.message || 'Your session has expired. Please refresh the page and try again.');
    }
  }
  return j;
}

function escapeHtml(s){
  return (s||'').replace(/[&<>"']/g,function(m){
    return {'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;','\'':'&#039;'}[m];
  });
}
function statusLabel(id){
  if (id==1) return 'Backlog';
  if (id==2) return 'Planned';
  if (id==3) return 'In Progress';
  if (id==4) return 'Released';
  if (id==5) return 'Rejected';
  return '';
}
function statusBadge(id){
  const label=statusLabel(id);
  if(!label) return '';
  return '<span class="badge status-badge-'+id+'">'+label+'</span>';
}

let CATS = [];
let FEATS = [];
let ROADMAP_FEATS = [];
const ROADMAP_STATUS_IDS = [1,2,3,4,5];

document.addEventListener('DOMContentLoaded', async ()=>{
  await getCsrf();

  const loginCard = document.getElementById('login-card');
  const adminUI   = document.getElementById('admin-ui');

  async function refreshAdminFlag(){
    const me = await api('?action=admin_me',{headers:{}});
    if (me.user && me.user.role==='admin'){
      loginCard.style.display='none';
      adminUI.style.display='';
      loadSettings();
      loadCats();
      loadFeats();
      loadUsers();
      loadRoadmap();
    }
  }
  await refreshAdminFlag();

  // Admin login
  document.getElementById('btn-adm-login').addEventListener('click', async ()=>{
    const id = document.getElementById('adm-ident').value.trim();
    const pw = document.getElementById('adm-pass').value;
    const msg = document.getElementById('adm-msg');
    const j = await api('?action=admin_login',{method:'POST',body:JSON.stringify({identifier:id,password:pw})});
    if (j.ok){
      msg.textContent='Logged in.';
      await refreshAdminFlag();
    } else {
      msg.textContent = j.error || 'Login failed.';
    }
  });

  // Logout
  document.getElementById('btn-logout').addEventListener('click', async ()=>{
    await api('?action=logout',{method:'POST',body:'{}'});
    window.location.href='admin.php';
  });

  // Tabs
  document.querySelectorAll('.tabbtn').forEach(btn=>{
    btn.addEventListener('click', ()=>{
      document.querySelectorAll('.tabbtn').forEach(b=>b.classList.remove('active'));
      document.querySelectorAll('.tab').forEach(t=>t.classList.remove('active'));
      btn.classList.add('active');
      document.getElementById('tab-'+btn.dataset.tab).classList.add('active');
    });
  });

  // SETTINGS --------------------
  async function loadSettings(){
    const j = await api('?action=settings_get',{headers:{}});
    document.getElementById('set-domain').value = j.allowed_domain || '';
    const limitInput = document.getElementById('set-limit');
    if (limitInput) {
      limitInput.value = (j.submission_limit ?? 0);
    }
  
    const voteLimitInput = document.getElementById('set-vote-limit');
    if (voteLimitInput) {
      voteLimitInput.value = (j.vote_limit ?? 0);
    }
}

  document.getElementById('btn-save-domain').addEventListener('click', async ()=>{
    const val = document.getElementById('set-domain').value.trim();
    const msg = document.getElementById('set-domain-msg');
    const j = await api('?action=settings_set_domain',{method:'POST',body:JSON.stringify({allowed_domain:val})});
    msg.textContent = j.ok ? 'Saved' : (j.error || 'Error');
  });

  const btnSaveLimit = document.getElementById('btn-save-limit');
  if (btnSaveLimit) {
    btnSaveLimit.addEventListener('click', async ()=>{
      const raw = document.getElementById('set-limit').value.trim();
      const msg = document.getElementById('set-limit-msg');
      const val = raw === '' ? 0 : Number(raw);
      const j = await api('?action=settings_set_limit', {
        method:'POST',
        body: JSON.stringify({ submission_limit: val })
      });
      msg.textContent = j.ok ? 'Saved' : (j.error || 'Error');
    });
  }

  const btnSaveVoteLimit = document.getElementById('btn-save-vote-limit');
  if (btnSaveVoteLimit) {
    btnSaveVoteLimit.addEventListener('click', async ()=>{
      const raw = document.getElementById('set-vote-limit').value.trim();
      const msg = document.getElementById('set-vote-limit-msg');
      const val = raw === '' ? 0 : Number(raw);
      const j = await api('?action=settings_set_vote_limit', {
        method:'POST',
        body: JSON.stringify({ vote_limit: val })
      });
      msg.textContent = j.ok ? 'Saved' : (j.error || 'Error');
    });
  }


  // CATEGORIES ------------------
  async function loadCats(){
    const cats = await api('?action=categories_list',{headers:{}});
    CATS = cats;
    const tbody = document.querySelector('#cat-table tbody');
    tbody.innerHTML = cats.map(c=>
      '<tr><td>'+escapeHtml(c.name)+'</td><td>'+c.sort_order+'</td>'+
      '<td><button class="btn btn-del-cat" data-id="'+c.id+'">Delete</button></td></tr>'
    ).join('');

    const filt = document.getElementById('filt-cat');
    filt.innerHTML = '<option value="">All categories</option>' +
      cats.map(c=>'<option value="'+c.id+'">'+escapeHtml(c.name)+'</option>').join('');

    document.querySelectorAll('.btn-del-cat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        if (!confirm('Delete this category?')) return;
        await api('?action=categories_delete',{method:'POST',body:JSON.stringify({id})});
        loadCats();
        loadRoadmap();
      });
    });

    // Fill admin "new feature" category select
    const newCat = document.getElementById('new-feat-cat');
    if (newCat) {
      newCat.innerHTML = '<option value="">(none)</option>' +
        cats.map(c => '<option value="'+c.id+'">'+escapeHtml(c.name)+'</option>').join('');
    }
  }

  document.getElementById('btn-add-cat').addEventListener('click', async ()=>{
    const name = document.getElementById('cat-name').value.trim();
    const sort = Number(document.getElementById('cat-sort').value || '100');
    if (!name) return;
    await api('?action=categories_create',{method:'POST',body:JSON.stringify({name,sort_order:sort})});
    document.getElementById('cat-name').value='';
    loadCats();
    loadRoadmap();
  });

  // FEATURES --------------------
  function renderFeatEditRow(id){
    const tbody = document.querySelector('#feat-table tbody');
    tbody.querySelectorAll('.feat-edit-row').forEach(r=>r.remove());
    const feat = FEATS.find(f=>f.id==id);
    if (!feat) return;
    const rows = Array.from(tbody.querySelectorAll('tr'));
    const anchor = rows.find(tr=>{
      const btn = tr.querySelector('.btn-edit-feat');
      return btn && Number(btn.dataset.id)===id;
    });
    if (!anchor) return;

    const tr = document.createElement('tr');
    tr.className = 'feat-edit-row';
    tr.dataset.id = id;
    const td = document.createElement('td');
    td.colSpan = 8;

    td.innerHTML = `
      <div class="edit-card">
        <div class="edit-row-grid">
          <div>
            <div class="mb8">
              <div class="muted">Title</div>
              <input class="edit-title" value="${escapeHtml(feat.title)}" />
            </div>
            <div class="mb8">
              <div class="muted">Description</div>
              <textarea class="edit-desc" rows="4">${escapeHtml(feat.description || '')}</textarea>
            </div>
            <div class="muted" style="font-size:12px;">
              Author: ${escapeHtml(feat.author_email || '')}<br>
              Created: ${feat.created_at} · Updated: ${feat.updated_at}<br>
              Votes: ${feat.total_votes || 0} ${feat.duplicate_of ? '· Duplicate of #'+feat.duplicate_of : ''}
            </div>
          </div>
          <div>
            <div class="mb8">
              <div class="muted">Status</div>
              <select class="edit-status">
                <option value="1" ${feat.status_id==1?'selected':''}>Backlog</option>
                <option value="2" ${feat.status_id==2?'selected':''}>Planned</option>
                <option value="3" ${feat.status_id==3?'selected':''}>In Progress</option>
                <option value="4" ${feat.status_id==4?'selected':''}>Released</option>
                <option value="5" ${feat.status_id==5?'selected':''}>Rejected</option>
              </select>
            </div>
            <div class="mb8">
              <div class="muted">Category</div>
              <select class="edit-cat">
                <option value="">(none)</option>
                ${CATS.map(c => `
                  <option value="${c.id}" ${c.id==feat.category_id?'selected':''}>
                    ${escapeHtml(c.name)}
                  </option>
                `).join('')}
              </select>
            </div>

            ${
              feat.rejection_reason
                ? `
                  <div class="mb8">
                    <div class="muted">Last rejection message</div>
                    <div style="font-size:12px;white-space:pre-wrap;">
                      ${escapeHtml(feat.rejection_reason)}
                    </div>
                  </div>
                `
                : ''
            }

            <div class="mb8">
              <button class="btn btn-secondary btn-save-edit" type="button">Save changes</button>
              <button class="btn btn-secondary btn-cancel-edit" type="button">Close</button>
            </div>
          </div>

        </div>
      </div>`;

    tr.appendChild(td);
    anchor.after(tr);

    tr.querySelector('.btn-cancel-edit').addEventListener('click', () => tr.remove());
    tr.querySelector('.btn-save-edit').addEventListener('click', async () => {
      const title    = tr.querySelector('.edit-title').value.trim();
      const desc     = tr.querySelector('.edit-desc').value.trim();
      const statusEl = tr.querySelector('.edit-status');
      const statusId = Number(statusEl.value);
      const catVal   = tr.querySelector('.edit-cat').value;

      // Detect transition → Rejected (5) and prompt for message
      const prevStatus = feat.status_id;
      let rejectionMessage = null;

      if (prevStatus !== 5 && statusId === 5) {
        const msg = window.prompt(
          'Enter a rejection message to email to the original submitter (optional).\n\n' +
          'Click Cancel to keep the current status.',
          ''
        );
        if (msg === null) {
          // Admin cancelled → revert dropdown and abort save
          statusEl.value = String(prevStatus);
          return;
        }
        rejectionMessage = msg.trim();
      }

      const payload = {
        id: id,
        title: title,
        description: desc,
        status_id: statusId,
        category_id: (catVal === '') ? null : Number(catVal)
      };

      if (rejectionMessage && rejectionMessage.length > 0) {
        payload.rejection_message = rejectionMessage;
      }

      await api('?action=admin_feature_update', {
        method: 'POST',
        body: JSON.stringify(payload)
      });

      await loadFeats();
      await loadRoadmap();
    });

  }

  async function loadFeats(){
    const status = document.getElementById('filt-status').value;
    const cat    = document.getElementById('filt-cat').value;
    const q      = document.getElementById('filt-q').value.trim();
    const params = [];
    if (status) params.push('status='+encodeURIComponent(status));
    if (cat) params.push('category_id='+encodeURIComponent(cat));
    if (q) params.push('q='+encodeURIComponent(q));
    const qs = params.length ? ('&'+params.join('&')) : '';
    const data = await api('?action=admin_features_list'+qs,{headers:{}});
    FEATS = data;
    const tbody = document.querySelector('#feat-table tbody');
    tbody.innerHTML = data.map(f=>{
      return '<tr>'+
        '<td>'+f.id+'</td>'+
        '<td>'+escapeHtml(f.title)+'</td>'+
        '<td>'+statusBadge(f.status_id)+'</td>'+
        '<td>'+(f.category_id||'')+'</td>'+
        '<td>'+(f.total_votes||0)+'</td>'+
        '<td>'+escapeHtml(f.author_email||'')+'</td>'+
        '<td>'+(f.duplicate_of||'')+'</td>'+
        '<td>'+
          '<button class="btn btn-edit-feat" data-id="'+f.id+'">Edit</button> '+
          '<button class="btn btn-dup-feat" data-id="'+f.id+'">Mark dup</button> '+
          '<button class="btn btn-del-feat" data-id="'+f.id+'">Del</button>'+
        '</td>'+
      '</tr>';
    }).join('');

    document.querySelectorAll('.btn-del-feat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        if (!confirm('Delete feature #'+id+'?')) return;
        await api('?action=admin_feature_delete',{method:'POST',body:JSON.stringify({id})});
        loadFeats();
        loadRoadmap();
      });
    });
    document.querySelectorAll('.btn-dup-feat').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        const dup = prompt('Mark feature #'+id+' as duplicate of feature id:');
        if (!dup) return;
        await api('?action=features_mark_duplicate',{method:'POST',body:JSON.stringify({feature_id:id,duplicate_of_id:Number(dup)})});
        loadFeats();
        loadRoadmap();
      });
    });
    document.querySelectorAll('.btn-edit-feat').forEach(b=>{
      b.addEventListener('click', ()=>{
        const id = Number(b.dataset.id);
        renderFeatEditRow(id);
      });
    });
  }
  document.getElementById('btn-filt').addEventListener('click', loadFeats);

  // -------- EXPORT FEATURES (CSV) ----------
  const btnExportFeats = document.getElementById('btn-export-feats');
  if (btnExportFeats) {
    btnExportFeats.addEventListener('click', () => {
      const status = document.getElementById('filt-status').value;
      const cat    = document.getElementById('filt-cat').value;
      const q      = document.getElementById('filt-q').value.trim();

      const params = [];
      if (status) params.push('status=' + encodeURIComponent(status));
      if (cat)    params.push('category_id=' + encodeURIComponent(cat));
      if (q)      params.push('q=' + encodeURIComponent(q));

      const qs = params.length ? '&' + params.join('&') : '';
      window.location.href = 'api.php?action=admin_features_export' + qs;
    });
  }

    // -------- IMPORT FEATURES (CSV) ----------
  const btnImportFeats  = document.getElementById('btn-import-feats');
  const importInput     = document.getElementById('feat-import-file');
  const importMsg       = document.getElementById('feat-import-msg');

  if (btnImportFeats && importInput) {
    // clicking the button opens file picker
    btnImportFeats.addEventListener('click', () => {
      importInput.click();
    });

    importInput.addEventListener('change', async () => {
      if (!importInput.files || !importInput.files.length) return;
      const file = importInput.files[0];

      if (importMsg) {
        importMsg.textContent = 'Uploading and importing...';
      }

      const fd = new FormData();
      fd.append('file', file);

      try {
        const res = await fetch('api.php?action=admin_features_import', {
          method: 'POST',
          headers: {
            'X-CSRF-Token': CSRF
            // no Content-Type here; browser sets multipart boundary
          },
          body: fd
        });

        // Read raw response first
        const raw = await res.text();
        let j = null;

        // Try JSON, but don't die if it fails
        try {
          j = JSON.parse(raw);
        } catch (e) {
          j = null;
        }

        if (!res.ok) {
          // Non-200: show error (JSON error if present, else first bit of raw)
          if (importMsg) {
            if (j && j.error) {
              importMsg.textContent = j.error;
            } else {
              importMsg.textContent =
                'Import failed: ' + (raw ? raw.substring(0, 200) : 'server error.');
            }
          }
          return;
        }

        // HTTP 200 -> treat as success even if response wasn't JSON
        const created =
          j && typeof j.created !== 'undefined'
            ? j.created
            : null;

        if (importMsg) {
          if (j && j.ok && created !== null) {
            importMsg.textContent = 'Imported ' + created + ' new feature(s).';
          } else if (j && j.ok) {
            importMsg.textContent = 'Import completed.';
          } else {
            // Non-JSON or no "ok" field but still 200
            importMsg.textContent = 'Import completed.';
          }
        }

        // 🔄 Immediately refresh Features + Roadmap
        await loadFeats();
        await loadRoadmap();

      } catch (err) {
        if (importMsg) {
          importMsg.textContent = 'Import failed: network or server error.';
        }
      } finally {
        importInput.value = '';
      }
    });
  }


  // Admin "create feature" inline
  const btnNewFeatSave  = document.getElementById('btn-new-feat-save');
  const btnNewFeatClear = document.getElementById('btn-new-feat-clear');
  const newFeatMsg      = document.getElementById('new-feat-msg');

  if (btnNewFeatSave) {
    btnNewFeatSave.addEventListener('click', async ()=>{
      const title  = document.getElementById('new-feat-title').value.trim();
      const desc   = document.getElementById('new-feat-desc').value.trim();
      const author = document.getElementById('new-feat-author').value.trim();
      const status = Number(document.getElementById('new-feat-status').value || 1);
      const catVal = document.getElementById('new-feat-cat').value;

      if (!title || !author) {
        newFeatMsg.textContent = 'Title and author email are required.';
        return;
      }

      const payload = {
        title,
        description: desc,
        author_email: author,
        status_id: status,
        category_id: (catVal === '') ? null : Number(catVal)
      };

      const j = await api('?action=admin_feature_create',{
        method:'POST',
        body: JSON.stringify(payload)
      });

      if (j.ok) {
        newFeatMsg.textContent = 'Created feature #'+j.id;
        document.getElementById('new-feat-title').value = '';
        document.getElementById('new-feat-desc').value = '';
        document.getElementById('new-feat-author').value = '';
        document.getElementById('new-feat-status').value = '1';
        document.getElementById('new-feat-cat').value = '';
        loadFeats();
        loadRoadmap();
      } else {
        newFeatMsg.textContent = j.error || 'Error creating feature.';
      }
    });
  }

  if (btnNewFeatClear) {
    btnNewFeatClear.addEventListener('click', ()=>{
      document.getElementById('new-feat-title').value = '';
      document.getElementById('new-feat-desc').value = '';
      document.getElementById('new-feat-author').value = '';
      document.getElementById('new-feat-status').value = '1';
      document.getElementById('new-feat-cat').value = '';
      if (newFeatMsg) newFeatMsg.textContent = '';
    });
  }

  // Toggle visibility of create-feature form
  const btnToggleNewFeat = document.getElementById('btn-toggle-new-feat');
  if (btnToggleNewFeat) {
    btnToggleNewFeat.addEventListener('click', () => {
      const card = document.getElementById('new-feat-card');
      if (!card) return;
      const isVisible = card.style.display !== 'none';
      card.style.display = isVisible ? 'none' : 'block';
      btnToggleNewFeat.textContent = isVisible ? 'Add feature' : 'Hide form';
    });
  }

  // ROADMAP ---------------------
  function renderRoadmapBoard(features){
    const byStatus = {1:[],2:[],3:[],4:[],5:[]};

    features.forEach(f => {
      if (!byStatus[f.status_id]) return;
      // If you ever want to hide dup children from the board, uncomment:
      // if (f.duplicate_of) return;
      byStatus[f.status_id].push(f);
    });

    ROADMAP_STATUS_IDS.forEach(st => {
      const body = document.getElementById('roadmap-col-'+st);
      if (!body) return;
      const list = byStatus[st] || [];
      body.innerHTML = list.map(f => `
        <div class="roadmap-card"
             draggable="true"
             data-id="${f.id}">
          <div class="roadmap-card-title">${escapeHtml(f.title)}</div>
          <div class="muted roadmap-card-meta">
            #${f.id}
            ${f.author_email ? ' · '+escapeHtml(f.author_email) : ''}
            · ${(f.total_votes || 0)} votes
          </div>
        </div>
      `).join('');
    });

    enableRoadmapDnD();
  }

  function enableRoadmapDnD(){
      // Cards are re-rendered every time, so it's fine to (re)attach handlers to them
      const cards = document.querySelectorAll('.roadmap-card');
      cards.forEach(card => {
        card.addEventListener('dragstart', e => {
          e.dataTransfer.setData('text/plain', card.dataset.id);
          card.classList.add('dragging');
        });
        card.addEventListener('dragend', () => {
          card.classList.remove('dragging');
        });
      });
    
      // Columns are persistent; only attach DnD handlers ONCE per column
      const columns = document.querySelectorAll('.roadmap-column-body');
      columns.forEach(col => {
        const colRoot = col.parentElement;
    
        // guard so we don't stack multiple listeners
        if (col.dataset.dndInit === '1') return;
        col.dataset.dndInit = '1';
    
        col.addEventListener('dragover', e => {
          e.preventDefault();
          colRoot.classList.add('drag-over');
        });
    
        col.addEventListener('dragleave', () => {
          colRoot.classList.remove('drag-over');
        });
    
        col.addEventListener('drop', async e => {
          e.preventDefault();
          colRoot.classList.remove('drag-over');
    
          const featureId = e.dataTransfer.getData('text/plain');
          const statusId  = Number(col.dataset.status);
          if (!featureId || !statusId) return;
    
          // Find current feature to know old status & title
          const feat = (ROADMAP_FEATS || []).find(f => String(f.id) === String(featureId));
          const oldStatus = feat ? Number(feat.status_id) : null;
    
          let payload = {
            id: Number(featureId),
            status_id: statusId
          };
    
          // If moving INTO Rejected (5), prompt once for a rejection message
          if (oldStatus !== 5 && statusId === 5) {
            const msg = window.prompt(
              'Enter a rejection message to email to the original submitter (optional).\n\n' +
              'Click Cancel to keep the current status.',
              ''
            );
            if (msg === null) {
              // Admin cancelled → do NOT change status
              return;
            }
            const trimmed = msg.trim();
            if (trimmed.length > 0) {
              payload.rejection_message = trimmed;
            }
          }
    
          await api('?action=admin_feature_update', {
            method: 'POST',
            body: JSON.stringify(payload)
          });
    
          await loadFeats();
          await loadRoadmap();
        });
      });
    }


  async function loadRoadmap(){
    // Roadmap should always see the full picture, ignore filters
    const data = await api('?action=admin_features_list',{headers:{}});
    ROADMAP_FEATS = data || [];
    renderRoadmapBoard(ROADMAP_FEATS);
  }

  // USERS -----------------------
  async function loadUsers(){
    const data = await api('?action=admin_users_list',{headers:{}});
    const tbody = document.querySelector('#user-table tbody');
    tbody.innerHTML = data.map(u=>{
      const tick = u.role==='admin' ? '✔' : '';
      const ls = u.last_seen_at ? new Date(u.last_seen_at.replace(' ','T')+'Z').toLocaleDateString() : '';
      return '<tr>'+
        '<td>'+escapeHtml(u.email)+'</td>'+
        '<td style="text-align:center">'+tick+'</td>'+
        '<td>'+ls+'</td>'+
        '<td>'+(u.cnt_backlog||0)+'</td>'+
        '<td>'+(u.cnt_planned||0)+'</td>'+
        '<td>'+(u.cnt_progress||0)+'</td>'+
        '<td>'+(u.cnt_released||0)+'</td>'+
        '<td>'+(u.cnt_rejected||0)+'</td>'+
        '<td>'+
          '<button class="btn btn-toggle-admin" data-email="'+escapeHtml(u.email)+'">Toggle admin</button> '+
          '<button class="btn btn-view-user" data-id="'+u.id+'">View</button> '+
          '<button class="btn btn-secondary btn-del-user" data-id="'+u.id+'">Delete</button>'+
        '</td>'+
      '</tr>';
    }).join('');

    // Toggle admin
    document.querySelectorAll('.btn-toggle-admin').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const email = b.dataset.email;
        const row = data.find(u=>u.email===email);
        if (!row) return;
        const newRole = row.role==='admin' ? 'user' : 'admin';
        await api('?action=users_set_role',{method:'POST',body:JSON.stringify({email,role:newRole})});
        loadUsers();
      });
    });

    // View user detail
    document.querySelectorAll('.btn-view-user').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        const info = await api('?action=admin_user_detail&id='+id,{headers:{}});
        const ud = document.getElementById('user-detail');
        const u = info.user;
        const feats = info.features || [];
        let html = '<h3>User detail</h3>';
        html += '<p><strong>'+escapeHtml(u.email)+'</strong> ('+escapeHtml(u.name||'')+') – '+(u.role==='admin'?'Admin':'Standard')+'</p>';
        html += '<p class="muted">Last seen: '+(u.last_seen_at||'n/a')+'</p>';
        if (!feats.length){
          html += '<p class="muted">No submissions yet.</p>';
        } else {
          html += '<table><thead><tr><th>ID</th><th>Title</th><th>Status</th><th>Created</th><th>Updated</th></tr></thead><tbody>';
          html += feats.map(f=>'<tr><td>'+f.id+'</td><td>'+escapeHtml(f.title)+'</td><td>'+statusLabel(f.status_id)+'</td><td>'+f.created_at+'</td><td>'+f.updated_at+'</td></tr>').join('');
          html += '</tbody></table>';
        }
        ud.innerHTML = html;
      });
    });

    // Delete user with optional reassignment of open features
    document.querySelectorAll('.btn-del-user').forEach(b=>{
      b.addEventListener('click', async ()=>{
        const id = Number(b.dataset.id);
        const row = data.find(u=>u.id===id);
        if (!row) return;

        const openCount = (row.cnt_backlog||0) + (row.cnt_planned||0) + (row.cnt_progress||0);
        let reassignToUserId = null;

        if (openCount > 0) {
          const admins = data.filter(u => u.role === 'admin' && u.id !== id);
          if (!admins.length) {
            alert('No other admins exist to reassign open features. Create another admin first.');
            return;
          }

          const list = admins.map(a=>a.email).join(', ');
          const email = prompt(
            'This user has '+openCount+' open features (Backlog / Planned / In Progress).\n' +
            'Enter the admin email to reassign them to:\n' + list
          );
          if (!email) return; // cancelled

          const target = admins.find(a => a.email === email.trim());
          if (!target) {
            alert('Admin not found in list.');
            return;
          }
          reassignToUserId = target.id;
        } else {
          if (!confirm('Delete this user?')) return;
        }

        await api('?action=users_delete_with_reassign',{
          method:'POST',
          body: JSON.stringify({
            user_id: id,
            reassign_to_user_id: reassignToUserId
          })
        });
        loadUsers();
        loadFeats();
        loadRoadmap();
      });
    });
  }

  // Create admin user
  document.getElementById('btn-create-admin').addEventListener('click', async ()=>{
    const email = document.getElementById('u-email').value.trim();
    const name  = document.getElementById('u-name').value.trim();
    const pass  = document.getElementById('u-pass').value;
    if (!email || !pass) return;
    await api('?action=users_create_admin',{method:'POST',body:JSON.stringify({email,name,password:pass})});
    document.getElementById('u-email').value='';
    document.getElementById('u-name').value='';
    document.getElementById('u-pass').value='';
    loadUsers();
  });

});
</script>
</body>
</html>
