<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Dry Blend Calculator</title>
  <style>
    body { font-family: sans-serif; padding: 20px; }
    .product-row { display: flex; gap: 10px; margin-bottom: 10px; flex-wrap: wrap; align-items:center; }
    .product-row input, .product-row select { padding: 5px; width: 160px; }
    .product-row input.rate-input { width: 120px; }
    .product-row input.price-input { width: 120px; }
    .remove-btn { color: red; cursor: pointer; font-weight: bold; padding: 0 8px; }
    .product-container { margin-bottom: 20px; }
    .totals { margin-top: 20px; font-weight: bold; }
    .btn { background: green; color: white; padding: 8px 12px; cursor: pointer; border: none; border-radius: 4px; }
    .print { background: blue; color: white; padding: 8px 12px; cursor: pointer; border: none; border-radius: 4px; }
    .clear-all { margin-left: 10px; background: red; }
    .blend-setup, .price-summary {margin-top: 20px; font-weight: bold;}
    .blend-setup input {padding: 5px; width: 100px; margin-left: 10px;}
    
    .pricing-unit-label{
      width: 90px;
      padding: 5px 0;
      color:#333;
    }


    .switch {
      position: relative;
      display: inline-block;
      width: 40px;
      height: 24px;
      margin-right: 8px;
    }
    .switch input { opacity: 0; width: 0; height: 0; }
    .slider {
      position: absolute;
      cursor: pointer;
      top: 0; left: 0; right: 0; bottom: 0;
      background-color: #ccc;
      transition: 0.4s;
      border-radius: 24px;
    }
    .slider:before {
      position: absolute;
      content: "";
      height: 18px; width: 18px;
      left: 3px; bottom: 3px;
      background-color: white;
      transition: 0.4s;
      border-radius: 50%;
    }
    input:checked + .slider { background-color: #4CAF50; }
    input:checked + .slider:before { transform: translateX(16px); }

    /* Print behavior */
    #printOnlyOutput { display: none; }
    @media print {
      #appRoot { display: none !important; }
      #printOnlyOutput { display: block !important; }
      body { padding: 0; }
    }
  </style>
</head>
<body>

<div id="appRoot">
  <h2>Dry Blend Calculator</h2>

  <div class="product-container" id="productContainer"></div>

  <button class="btn" id="addBtn" type="button">+ Add Product</button>
  <button class="btn clear-all" id="clearBtn" type="button">Clear All</button>
  <button class="print" id="printBtn" type="button" title="Print Report">Print</button>

  <div class="totals">
    <p><strong>Units/Total Product (lbs):</strong></p>
    <p>
      N: <span id="totalN">0</span> |
      P: <span id="totalP">0</span> |
      K: <span id="totalK">0</span> |
      Ca: <span id="totalCa">0</span> |
      S: <span id="totalS">0</span> |
      Mg: <span id="totalMg">0</span> |
      Zn: <span id="totalZn">0</span> |
      Fe: <span id="totalFe">0</span> |
      Mn: <span id="totalMn">0</span> |
      Cu: <span id="totalCu">0</span> |
      B: <span id="totalB">0</span>
    </p>
  </div>

  <div class="blend-setup">
    <label>Blend Fee ($/ton):</label>
    <input type="number" id="blendFee" value="0" />
  </div>

  <div class="price-summary">
    <p>Including Blend Fee: <span id="priceTotal">0.00</span> $/Total Product</p>
    <p>Including Blend Fee: <span id="pricePerTon">0.00</span> $/Ton</p>
  </div>

  <div class="preferences-summary" style="margin-top: 20px; display: flex; gap: 40px; flex-wrap: wrap;">
    <div class="prefs">
      <strong>Preferences:</strong><br />
      Blend Type: <span id="blendType">Dry</span>
    </div>

    <div class="summary">
      <strong>Blend Summary:</strong><br />
      Total Product: <span id="totalProduct">0</span> lbs<br />
      Blend Density: <span id="blendDensity">0</span> lbs/cubic foot<br />
      Blend Weight: <span id="blendWeight">0</span> ton
    </div>
  </div>

  <div class="nutrient-totals" style="margin-top: 20px; display: flex; gap: 40px; flex-wrap: wrap;">
    <div class="units">
      <strong>Units/Total Product (lbs):</strong><br />
      N: <span id="uN">0</span><br />
      P: <span id="uP">0</span><br />
      K: <span id="uK">0</span><br />
      Ca: <span id="uCa">0</span><br />
      S: <span id="uS">0</span><br />
    </div>

    <div class="analysis">
      <strong>Blend Analysis (%):</strong><br />
      N: <span id="aN">0</span><br />
      P: <span id="aP">0</span><br />
      K: <span id="aK">0</span><br />
      Ca: <span id="aCa">0</span><br />
      S: <span id="aS">0</span><br />
    </div>

    <div class="print-toggle" style="margin-top: 20px;">
      <label class="switch">
        <input type="checkbox" id="showPricesToggle" />
        <span class="slider round"></span>
      </label>
      <span>Show Individual Product Price</span>
    </div>
  </div>
</div>

<div id="printOnlyOutput" style="display:none;"></div>

<script>
  let productData = [];
  let printLock = false;

  async function loadProductData() {
    const response = await fetch('load_products.php');
    productData = await response.json();

    document.getElementById('addBtn').disabled = false;
    document.getElementById('clearBtn').disabled = false;

    addProductRow();
  }

  function togglePriceDisplay() {
    // NOTE:
    // Pricing inputs ($/ton) must ALWAYS be visible on the calculator screen.
    // This toggle ONLY controls the PRINT layout (row-level pricing columns),
    // handled inside generatePrintLayout().
  }

  function addProductRow() {
    if (productData.length === 0) {
      alert("Product data not loaded yet.");
      return;
    }

    const container = document.getElementById('productContainer');
    const row = document.createElement('div');
    row.className = 'product-row';

    // Dropdown
    const select = document.createElement('select');

    const defaultOption = document.createElement('option');
    defaultOption.value = '';
    defaultOption.textContent = 'Choose ingredient...';
    defaultOption.disabled = true;
    defaultOption.selected = true;
    select.appendChild(defaultOption);

    productData.forEach(product => {
      if ((product.productType || '').toLowerCase() === 'dry') {
        const option = document.createElement('option');
        option.value = product.productId;
        option.textContent = product.productAlias + ' — ' + product.productName;
        select.appendChild(option);
      }
    });
    select.addEventListener('change', updateTotals);

    // Lbs/acre input
    const lbsAcre = document.createElement('input');
    lbsAcre.type = 'number';
    lbsAcre.placeholder = 'lbs/acre';
    lbsAcre.className = 'rate-input';
    lbsAcre.addEventListener('input', updateTotals);

    // Price per ton input (ALWAYS visible)
    const priceInput = document.createElement('input');
    priceInput.type = 'number';
    priceInput.placeholder = '$/ton';
    priceInput.className = 'price-input';
    priceInput.addEventListener('input', updateTotals);
    
    // Pricing unit display (text)
    const pricingUnitLabel = document.createElement('div');
    pricingUnitLabel.className = 'pricing-unit-label';
    pricingUnitLabel.textContent = ''; // will fill after product selection


    // Remove button
    const remove = document.createElement('span');
    remove.textContent = 'X';
    remove.className = 'remove-btn';
    remove.addEventListener('click', () => {
      row.remove();
      updateTotals();
    });

    row.appendChild(select);
    row.appendChild(lbsAcre);
    row.appendChild(priceInput);
    row.appendChild(pricingUnitLabel);
    row.appendChild(remove);
    container.appendChild(row);
  }

  function clearAllRows() {
    document.getElementById('productContainer').innerHTML = '';
    updateTotals();
  }

  function updateTotals() {
    let totals = {
      N: 0, P: 0, K: 0, Ca: 0, S: 0, Mg: 0, Zn: 0,
      Fe: 0, Mn: 0, Cu: 0, B: 0
    };
    
  function normUnit(u){
      return (u ?? '').toString().trim().toLowerCase();
    }
    
    // This is your “unitPrice” column (price unit) coming from CSV.
    // If your JSON key is literally unitPrice, this is correct.
    // If your key is pricingUnit or priceUnit, adjust here.
    function getPriceUnit(p){
      return normUnit(p.unitPrice || p.pricingUnit || p.priceUnit || p.unit || '');
    }


    const rows = document.querySelectorAll('.product-row');

    // Nutrient totals
    rows.forEach(row => {
      const select = row.querySelector('select');
      const rateEl = row.querySelector('.rate-input');
      const product = productData.find(p => String(p.productId) === String(select.value));
      const rate = parseFloat(rateEl?.value) || 0;

      if (product) {
        totals.N += (product.productN || 0) / 100 * rate;
        totals.P += (product.productP || 0) / 100 * rate;
        totals.K += (product.productK || 0) / 100 * rate;
        totals.Ca += (product.productCa || 0) / 100 * rate;
        totals.S += (product.productS || 0) / 100 * rate;
        totals.Mg += (product.productMg || 0) / 100 * rate;
        totals.Zn += (product.productZn || 0) / 100 * rate;
        totals.Fe += (product.productFe || 0) / 100 * rate;
        totals.Mn += (product.productMn || 0) / 100 * rate;
        totals.Cu += (product.productCu || 0) / 100 * rate;
        totals.B  += (product.productB  || 0) / 100 * rate;
      }
    });

    // Update UI totals
    document.getElementById('totalN').textContent = totals.N.toFixed(2);
    document.getElementById('totalP').textContent = totals.P.toFixed(2);
    document.getElementById('totalK').textContent = totals.K.toFixed(2);
    document.getElementById('totalCa').textContent = totals.Ca.toFixed(2);
    document.getElementById('totalS').textContent = totals.S.toFixed(2);
    document.getElementById('totalMg').textContent = totals.Mg.toFixed(2);
    document.getElementById('totalZn').textContent = totals.Zn.toFixed(2);
    document.getElementById('totalFe').textContent = totals.Fe.toFixed(2);
    document.getElementById('totalMn').textContent = totals.Mn.toFixed(2);
    document.getElementById('totalCu').textContent = totals.Cu.toFixed(2);
    document.getElementById('totalB').textContent  = totals.B.toFixed(2);

    // Pricing
    let blendFee = parseFloat(document.getElementById('blendFee').value) || 0;
    let totalCost = 0;
    let totalWeightTons = 0;

    rows.forEach(row => {
      const rateEl = row.querySelector('.rate-input');
      const priceEl = row.querySelector('.price-input');

      const rate = parseFloat(rateEl?.value) || 0; // lbs
      const pricePerTon = parseFloat(priceEl?.value) || 0;

      const productWeightTons = rate / 2000;
      const productCost = productWeightTons * pricePerTon;

      totalWeightTons += productWeightTons;
      totalCost += productCost;
    });

    const pricePerTotal = totalCost + (totalWeightTons * blendFee);
    const pricePerTon = totalWeightTons ? (pricePerTotal / totalWeightTons) : 0;

    document.getElementById('priceTotal').textContent = pricePerTotal.toFixed(2);
    document.getElementById('pricePerTon').textContent = pricePerTon.toFixed(2);

    // Blend Summary
    let totalLbs = 0;
    let weightedDensitySum = 0;

    rows.forEach(row => {
      const select = row.querySelector('select');
      const rateEl = row.querySelector('.rate-input');

      const product = productData.find(p => String(p.productId) === String(select.value));
      const rate = parseFloat(rateEl?.value) || 0;

      if (product) {
        totalLbs += rate;
        weightedDensitySum += rate * (product.productDensity || 0);
      }
    });

    const avgDensity = totalLbs > 0 ? (weightedDensitySum / totalLbs) : 0;
    const totalTons = totalLbs / 2000;

    document.getElementById('totalProduct').textContent = totalLbs.toFixed(2);
    document.getElementById('blendDensity').textContent = avgDensity.toFixed(2);
    document.getElementById('blendWeight').textContent = totalTons.toFixed(2);

    // Units + Analysis
    document.getElementById('uN').textContent = totals.N.toFixed(2);
    document.getElementById('uP').textContent = totals.P.toFixed(2);
    document.getElementById('uK').textContent = totals.K.toFixed(2);
    document.getElementById('uCa').textContent = totals.Ca.toFixed(2);
    document.getElementById('uS').textContent = totals.S.toFixed(2);

    document.getElementById('aN').textContent = totalLbs ? (totals.N / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aP').textContent = totalLbs ? (totals.P / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aK').textContent = totalLbs ? (totals.K / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aCa').textContent = totalLbs ? (totals.Ca / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aS').textContent = totalLbs ? (totals.S / totalLbs * 100).toFixed(1) : '0';
  }

  function generatePrintLayout() {
    const showPrices = document.getElementById('showPricesToggle').checked;
    const rows = document.querySelectorAll('.product-row');

    let html = `
      <div style="font-family: sans-serif; padding: 20px;">
        <img src="helena-logo.png" alt="Helena Logo" style="height:50px;"><br/><br/>
        <strong>Grower:</strong> ___________________________<br/>
        <strong>Field(s):</strong> ___________________________<br/><br/>
        <table border="1" cellspacing="0" cellpadding="6" width="100%">
          <thead>
            <tr>
              <th>Product</th>
              <th>lbs/acre</th>
              <th>Density</th>
    `;

    if (showPrices) {
      html += `<th>$ / ton</th><th>Est. Cost</th>`;
    }

    html += `</tr></thead><tbody>`;

    rows.forEach(row => {
      const select = row.querySelector('select');
      const rateInput = row.querySelector('.rate-input');
      const priceInput = row.querySelector('.price-input');

      const product = productData.find(p => String(p.productId) === String(select.value));
      const rate = parseFloat(rateInput?.value) || 0;
      const pricePerTon = parseFloat(priceInput?.value) || 0;
      const estCost = (rate / 2000) * pricePerTon;

      if (product && rate > 0) {
        html += `<tr>
          <td>${product.productAlias}<br/><small>${product.productName}</small></td>
          <td>${rate}</td>
          <td>${product.productDensity || 0}</td>`;
        if (showPrices) {
          html += `<td>$${pricePerTon.toFixed(2)}</td><td>$${estCost.toFixed(2)}</td>`;
        }
        html += `</tr>`;
      }
    });

    html += `</tbody></table><br/>`;

    html += `
      <table border="1" cellpadding="6" width="100%" style="margin-top: 20px;">
        <tr><th colspan="5">Units/Total Product (lbs)</th></tr>
        <tr><td>N</td><td>P</td><td>K</td><td>Ca</td><td>S</td></tr>
        <tr>
          <td>${document.getElementById('uN').textContent}</td>
          <td>${document.getElementById('uP').textContent}</td>
          <td>${document.getElementById('uK').textContent}</td>
          <td>${document.getElementById('uCa').textContent}</td>
          <td>${document.getElementById('uS').textContent}</td>
        </tr>
        <tr><th colspan="5">Blend Analysis (%)</th></tr>
        <tr><td>N</td><td>P</td><td>K</td><td>Ca</td><td>S</td></tr>
        <tr>
          <td>${document.getElementById('aN').textContent}</td>
          <td>${document.getElementById('aP').textContent}</td>
          <td>${document.getElementById('aK').textContent}</td>
          <td>${document.getElementById('aCa').textContent}</td>
          <td>${document.getElementById('aS').textContent}</td>
        </tr>
      </table><br/>

      <table border="0" cellpadding="6" width="100%" style="margin-top: 20px;">
        <tr>
          <td>
            <strong>Blend Summary:</strong><br/>
            Total Product: ${document.getElementById('totalProduct').textContent} lbs<br/>
            Blend Density: ${document.getElementById('blendDensity').textContent} lbs/cubic foot<br/>
            Blend Weight: ${document.getElementById('blendWeight').textContent} ton
          </td>
          <td>
            <strong>Price Summary:</strong><br/>
            $/Total Product: ${document.getElementById('priceTotal').textContent}<br/>
            $/Ton: ${document.getElementById('pricePerTon').textContent}
          </td>
        </tr>
      </table><br/>

      <div style="text-align:center; font-size:10px;">
        Created by: Chris Givens<br/>
        <span style="color:red;">Pricing and Blend Analysis are Subject to Change</span><br/>
        Printed on: ${new Date().toLocaleDateString()}
      </div>
    </div>
    `;

    document.getElementById('printOnlyOutput').innerHTML = html;
  }

  function doPrint() {
    if (printLock) return;
    printLock = true;

    updateTotals();
    generatePrintLayout();

    // use setTimeout 0 to let DOM update before print dialog
    setTimeout(() => {
      window.print();
      // unlock shortly after print dialog opens
      setTimeout(() => { printLock = false; }, 500);
    }, 0);
  }

  window.addEventListener('load', () => {
    loadProductData();

    document.getElementById('addBtn').addEventListener('click', addProductRow);
    document.getElementById('clearBtn').addEventListener('click', clearAllRows);
    document.getElementById('blendFee').addEventListener('input', updateTotals);

    document.getElementById('showPricesToggle').addEventListener('change', () => {
      togglePriceDisplay();
    });

    document.getElementById('printBtn').addEventListener('click', doPrint);

    togglePriceDisplay();
  });
</script>


</body>
</html>
