<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>Liquid Blend Calculator</title>
  <style>
    body { font-family: sans-serif; padding: 20px; }
    .product-row { display: flex; gap: 10px; margin-bottom: 10px; flex-wrap: wrap; align-items: center; }
    .product-row input, .product-row select { padding: 5px; width: 140px; }
    .remove-btn { color: red; cursor: pointer; font-weight: bold; margin-left: 6px; }
    .product-container { margin-bottom: 20px; }

    .totals { margin-top: 20px; font-weight: bold; }
    .btn { background: green; color: white; padding: 8px 12px; cursor: pointer; border: none; border-radius: 4px; }
    .print { background: blue; color: white; padding: 8px 12px; cursor: pointer; border: none; border-radius: 4px; }
    .clear-all { margin-left: 10px; background: red; }

    .blend-setup, .price-summary { margin-top: 20px; font-weight: bold; }
    .blend-setup input { padding: 5px; width: 110px; margin-left: 10px; }

    /* Toggle */
    .switch {
      position: relative; display: inline-block;
      width: 40px; height: 24px; margin-right: 8px;
    }
    .switch input { opacity: 0; width: 0; height: 0; }
    .slider {
      position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0;
      background-color: #ccc; transition: 0.4s; border-radius: 24px;
    }
    .slider:before {
      position: absolute; content: ""; height: 18px; width: 18px; left: 3px; bottom: 3px;
      background-color: white; transition: 0.4s; border-radius: 50%;
    }
    input:checked + .slider { background-color: #4CAF50; }
    input:checked + .slider:before { transform: translateX(16px); }

    /* Print behavior */
    #printOnlyOutput { display: none; }
    @media print {
      #appRoot { display: none !important; }
      #printOnlyOutput { display: block !important; }
      body { padding: 0; }
    }
  </style>
</head>

<body>
<div id="appRoot">
  <h2>Liquid Blend Calculator</h2>

  <div class="product-container" id="productContainer"></div>

  <button class="btn" id="addBtn" type="button" disabled>+ Add Product</button>
  <button class="btn clear-all" id="clearBtn" type="button" disabled>Clear All</button>
  <button class="print" id="printBtn" type="button" title="Print Report">Print</button>

  <div class="totals">
    <p><strong>Units/Total Product (lbs):</strong></p>
    <p>
      N: <span id="totalN">0</span> |
      P: <span id="totalP">0</span> |
      K: <span id="totalK">0</span> |
      Ca: <span id="totalCa">0</span> |
      S: <span id="totalS">0</span> |
      Mg: <span id="totalMg">0</span> |
      Zn: <span id="totalZn">0</span> |
      Fe: <span id="totalFe">0</span> |
      Mn: <span id="totalMn">0</span> |
      Cu: <span id="totalCu">0</span> |
      B: <span id="totalB">0</span>
    </p>
  </div>

  <div class="blend-setup">
    <label>Blend Fee ($/ton):</label>
    <input type="number" id="blendFee" value="0" />
  </div>

  <div class="price-summary">
    <p>Including Blend Fee: <span id="priceTotal">0.00</span> $/Total Product</p>
    <p>Including Blend Fee: <span id="pricePerTon">0.00</span> $/Ton</p>
  </div>

  <div class="preferences-summary" style="margin-top: 20px; display: flex; gap: 40px; flex-wrap: wrap;">
    <div class="prefs">
      <strong>Preferences:</strong><br />
      Blend Type: <span id="blendType">Liquid</span>
    </div>

    <div class="summary">
      <strong>Blend Summary:</strong><br />
      Total Product: <span id="totalProduct">0</span> lbs<br />
      Blend Density: <span id="blendDensity">0</span> lbs/gal<br />
      Blend Weight: <span id="blendWeight">0</span> ton
    </div>
  </div>

  <div class="nutrient-totals" style="margin-top: 20px; display: flex; gap: 40px; flex-wrap: wrap;">
    <div class="units">
      <strong>Units/Total Product (lbs):</strong><br />
      N: <span id="uN">0</span><br />
      P: <span id="uP">0</span><br />
      K: <span id="uK">0</span><br />
      Ca: <span id="uCa">0</span><br />
      S: <span id="uS">0</span><br />
    </div>

    <div class="analysis">
      <strong>Blend Analysis (%):</strong><br />
      N: <span id="aN">0</span><br />
      P: <span id="aP">0</span><br />
      K: <span id="aK">0</span><br />
      Ca: <span id="aCa">0</span><br />
      S: <span id="aS">0</span><br />
    </div>

    <div class="print-toggle" style="margin-top: 20px;">
      <label class="switch">
        <input type="checkbox" id="showPricesToggle" />
        <span class="slider round"></span>
      </label>
      <span>Show Individual Product Price (Print Only)</span>
    </div>
  </div>
</div>

<div id="printOnlyOutput"></div>

<script>
  let productData = [];

  function normStr(v) { return (v ?? '').toString().trim().toLowerCase(); }

  // STRICT: only productType === 'liquid'
  function isLiquidProduct(p) {
    return normStr(p.productType) === 'liquid';
  }

  async function loadProductData() {
    const response = await fetch('load_products.php');
    productData = await response.json();

    document.getElementById('addBtn').disabled = false;
    document.getElementById('clearBtn').disabled = false;

    addProductRow();
  }

  function addProductRow() {
    if (!Array.isArray(productData) || productData.length === 0) {
      alert("Product data not loaded yet.");
      return;
    }

    const container = document.getElementById('productContainer');
    const row = document.createElement('div');
    row.className = 'product-row';

    const select = document.createElement('select');
    const defaultOption = document.createElement('option');
    defaultOption.value = '';
    defaultOption.textContent = 'Choose liquid ingredient...';
    defaultOption.disabled = true;
    defaultOption.selected = true;
    select.appendChild(defaultOption);

    // Populate ONLY liquid items
    productData.forEach(p => {
      if (isLiquidProduct(p)) {
        const option = document.createElement('option');
        option.value = p.productId;
        option.textContent = (p.productAlias || '') + ' — ' + (p.productName || '');
        select.appendChild(option);
      }
    });

    select.addEventListener('change', updateTotals);

    // gal/acre
    const rate = document.createElement('input');
    rate.type = 'number';
    rate.placeholder = 'gal/acre';
    rate.className = 'rate-input'; // liquid: this means gallons
    rate.addEventListener('input', updateTotals);

    // $/gal
    const price = document.createElement('input');
    price.type = 'number';
    price.placeholder = '$/gal';
    price.className = 'price-input';
    price.addEventListener('input', updateTotals);

    const remove = document.createElement('span');
    remove.textContent = 'X';
    remove.className = 'remove-btn';
    remove.addEventListener('click', () => {
      row.remove();
      updateTotals();
    });

    row.appendChild(select);
    row.appendChild(rate);
    row.appendChild(price);
    row.appendChild(remove);
    container.appendChild(row);

    updateTotals();
  }

  function clearAllRows() {
    document.getElementById('productContainer').innerHTML = '';
    updateTotals();
  }

  function updateTotals() {
    let totals = { N:0,P:0,K:0,Ca:0,S:0,Mg:0,Zn:0,Fe:0,Mn:0,Cu:0,B:0 };
    const rows = document.querySelectorAll('.product-row');

    const blendFee = parseFloat(document.getElementById('blendFee').value) || 0;

    // For summary + pricing
    let totalLbs = 0;
    let totalTons = 0;
    let totalCost = 0;          // $/acre (sum of each product cost/acre)
    let weightedDensitySum = 0; // lbs-weighted density sum

    rows.forEach(row => {
      const select = row.querySelector('select');
      const galEl = row.querySelector('.rate-input');   // gal/acre
      const priceEl = row.querySelector('.price-input'); // $/gal

      const p = productData.find(x => String(x.productId) === String(select.value));
      if (!p) return;

      const gal = parseFloat(galEl?.value) || 0;
      const pricePerGal = parseFloat(priceEl?.value) || 0;

      const dens = parseFloat(p.productDensity || 0) || 0; // lbs/gal
      if (gal <= 0 || dens <= 0) return;

      // Convert liquid gal -> lbs so the rest of the calculator stays in lbs like dry
      const lbs = gal * dens;

      // Nutrient lbs/ac = (%/100) * lbs product/ac
      totals.N  += (p.productN  || 0) / 100 * lbs;
      totals.P  += (p.productP  || 0) / 100 * lbs;
      totals.K  += (p.productK  || 0) / 100 * lbs;
      totals.Ca += (p.productCa || 0) / 100 * lbs;
      totals.S  += (p.productS  || 0) / 100 * lbs;
      totals.Mg += (p.productMg || 0) / 100 * lbs;
      totals.Zn += (p.productZn || 0) / 100 * lbs;
      totals.Fe += (p.productFe || 0) / 100 * lbs;
      totals.Mn += (p.productMn || 0) / 100 * lbs;
      totals.Cu += (p.productCu || 0) / 100 * lbs;
      totals.B  += (p.productB  || 0) / 100 * lbs;

      totalLbs += lbs;
      const tons = lbs / 2000;
      totalTons += tons;

      // cost/acre = gal/acre * $/gal
      totalCost += gal * pricePerGal;

      // Weighted blend density (lbs/gal)
      weightedDensitySum += lbs * dens;
    });

    // Update top totals row
    for (const k of Object.keys(totals)) {
      const el = document.getElementById('total'+k);
      if (el) el.textContent = totals[k].toFixed(2);
    }

    // Price summary: include blend fee ($/ton)
    const priceTotal = totalCost + (totalTons * blendFee);
    const pricePerTon = totalTons ? (priceTotal / totalTons) : 0;

    document.getElementById('priceTotal').textContent = priceTotal.toFixed(2);
    document.getElementById('pricePerTon').textContent = pricePerTon.toFixed(2);

    // Blend summary
    const avgDensity = totalLbs ? (weightedDensitySum / totalLbs) : 0;

    document.getElementById('totalProduct').textContent = totalLbs.toFixed(2);
    document.getElementById('blendDensity').textContent = avgDensity.toFixed(2);
    document.getElementById('blendWeight').textContent = totalTons.toFixed(2);

    // Units/Total + Analysis
    document.getElementById('uN').textContent = totals.N.toFixed(2);
    document.getElementById('uP').textContent = totals.P.toFixed(2);
    document.getElementById('uK').textContent = totals.K.toFixed(2);
    document.getElementById('uCa').textContent = totals.Ca.toFixed(2);
    document.getElementById('uS').textContent = totals.S.toFixed(2);

    document.getElementById('aN').textContent  = totalLbs ? (totals.N  / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aP').textContent  = totalLbs ? (totals.P  / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aK').textContent  = totalLbs ? (totals.K  / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aCa').textContent = totalLbs ? (totals.Ca / totalLbs * 100).toFixed(1) : '0';
    document.getElementById('aS').textContent  = totalLbs ? (totals.S  / totalLbs * 100).toFixed(1) : '0';
  }

  function generatePrintLayout() {
    // Toggle affects ONLY row-level pricing columns in print
    const showPrices = document.getElementById('showPricesToggle').checked;
    const rows = document.querySelectorAll('.product-row');

    let html = `
      <div style="font-family:sans-serif; padding:20px;">
        <div style="text-align:center; margin-bottom:10px;">
          <img src="helena-logo.png" alt="Helena Logo" style="height:60px;">
        </div>

        <div style="margin-bottom:10px;">
          <strong>Grower:</strong> ___________________________<br>
          <strong>Field(s):</strong> ___________________________
        </div>

        <table border="1" cellspacing="0" cellpadding="6" width="100%" style="border-collapse:collapse;">
          <thead>
            <tr>
              <th>Product</th>
              <th>gal/acre</th>
              <th>Density (lbs/gal)</th>
    `;

    if (showPrices) {
      html += `<th>$ / gal</th><th>Est. Cost</th>`;
    }

    html += `</tr></thead><tbody>`;

    rows.forEach(row => {
      const select = row.querySelector('select');
      const galEl = row.querySelector('.rate-input');
      const priceEl = row.querySelector('.price-input');

      const p = productData.find(x => String(x.productId) === String(select.value));
      const gal = parseFloat(galEl?.value) || 0;
      const pricePerGal = parseFloat(priceEl?.value) || 0;
      const dens = parseFloat(p?.productDensity || 0) || 0;

      if (!p || gal <= 0) return;

      const estCost = gal * pricePerGal;

      html += `<tr>
        <td>${p.productAlias || ''}<br><small>${p.productName || ''}</small></td>
        <td>${gal}</td>
        <td>${dens || 0}</td>`;

      if (showPrices) {
        html += `<td>$${pricePerGal.toFixed(2)}</td><td>$${estCost.toFixed(2)}</td>`;
      }

      html += `</tr>`;
    });

    html += `</tbody></table>

      <br>

      <table border="1" cellpadding="6" width="100%" style="border-collapse:collapse;">
        <tr><th colspan="5">Units/Total Product (lbs)</th></tr>
        <tr><td>N</td><td>P</td><td>K</td><td>Ca</td><td>S</td></tr>
        <tr>
          <td>${document.getElementById('uN').textContent}</td>
          <td>${document.getElementById('uP').textContent}</td>
          <td>${document.getElementById('uK').textContent}</td>
          <td>${document.getElementById('uCa').textContent}</td>
          <td>${document.getElementById('uS').textContent}</td>
        </tr>
        <tr><th colspan="5">Blend Analysis (%)</th></tr>
        <tr><td>N</td><td>P</td><td>K</td><td>Ca</td><td>S</td></tr>
        <tr>
          <td>${document.getElementById('aN').textContent}</td>
          <td>${document.getElementById('aP').textContent}</td>
          <td>${document.getElementById('aK').textContent}</td>
          <td>${document.getElementById('aCa').textContent}</td>
          <td>${document.getElementById('aS').textContent}</td>
        </tr>
      </table>

      <br>

      <table border="0" cellpadding="6" width="100%">
        <tr>
          <td>
            <strong>Blend Summary:</strong><br>
            Total Product: ${document.getElementById('totalProduct').textContent} lbs<br>
            Blend Density: ${document.getElementById('blendDensity').textContent} lbs/gal<br>
            Blend Weight: ${document.getElementById('blendWeight').textContent} ton
          </td>
          <td>
            <strong>Price Summary:</strong><br>
            $/Total Product: ${document.getElementById('priceTotal').textContent}<br>
            $/Ton: ${document.getElementById('pricePerTon').textContent}
          </td>
        </tr>
      </table>

      <div style="text-align:center; font-size:10px; margin-top:16px;">
        Created by: Chris Givens<br>
        <span style="color:red;">Pricing and Blend Analysis are Subject to Change</span><br>
        Printed on: ${new Date().toLocaleDateString()}
      </div>
    </div>`;

    document.getElementById('printOnlyOutput').innerHTML = html;
  }

  function doPrint() {
    updateTotals();
    generatePrintLayout();
    requestAnimationFrame(() => requestAnimationFrame(() => window.print()));
  }

  window.addEventListener('load', () => {
    loadProductData();

    document.getElementById('addBtn').addEventListener('click', addProductRow);
    document.getElementById('clearBtn').addEventListener('click', clearAllRows);
    document.getElementById('blendFee').addEventListener('input', updateTotals);
    document.getElementById('printBtn').addEventListener('click', doPrint);

    // Print toggle only affects print columns
    document.getElementById('showPricesToggle').addEventListener('change', () => {});
  });
</script>
</body>
</html>
