<?php
header('Content-Type: application/json; charset=utf-8');

// OPTIONAL TEMP DEBUG (uncomment for one test, then re-comment)
// error_reporting(E_ALL);
// ini_set('display_errors', 1);

$csvFile = __DIR__ . '/resources/products.csv';
if (!file_exists($csvFile)) {
  echo json_encode(['error' => 'CSV file not found', 'path' => $csvFile]);
  exit;
}

$handle = fopen($csvFile, 'r');
if (!$handle) {
  echo json_encode(['error' => 'Could not open CSV']);
  exit;
}

// ---- helpers ----
function norm_str($v) {
  if ($v === null) return '';
  $v = (string)$v;

  // Strip UTF-8 BOM if present
  $v = preg_replace('/^\xEF\xBB\xBF/', '', $v);

  // Try iconv if available (does NOT require mbstring)
  if (function_exists('iconv')) {
    $converted = @iconv('Windows-1252', 'UTF-8//IGNORE', $v);
    if ($converted !== false && $converted !== '') {
      $v = $converted;
    }
  }

  return trim($v);
}

function to_float($v) {
  $v = norm_str($v);
  if ($v === '') return 0.0;
  $v = str_replace(',', '', $v);
  return is_numeric($v) ? (float)$v : 0.0;
}

// Read header row
$header = fgetcsv($handle);
if (!$header) {
  fclose($handle);
  echo json_encode(['error' => 'CSV header missing/invalid']);
  exit;
}

$header = array_map('norm_str', $header);

$products = [];

while (($row = fgetcsv($handle)) !== false) {
  if (count($row) === 1 && norm_str($row[0]) === '') continue;
  if (count($row) !== count($header)) continue;

  $row = array_map('norm_str', $row);
  $p = array_combine($header, $row);
  if (!$p) continue;

  $products[] = [
    'productId'   => norm_str($p['productId'] ?? ''),
    'productAlias'=> norm_str($p['productAlias'] ?? ''),
    'productName' => norm_str($p['productName'] ?? ''),
    'productType' => norm_str($p['productType'] ?? ''),

    // densities / ratios
    'productDensity' => to_float($p['productDensity'] ?? 0),
    'altDensity'     => to_float($p['altDensity'] ?? 0),
    'blendRatio'     => to_float($p['blendRatio'] ?? 0),

    // pricing + unit label for pricing
    'pricePerUnit' => to_float($p['pricePerUnit'] ?? 0),
    'blendPrice'   => to_float($p['blendPrice'] ?? 0),
    'unitPrice'    => norm_str($p['unitPrice'] ?? ''),

    // nutrients
    'productN'  => to_float($p['productN']  ?? 0),
    'productP'  => to_float($p['productP']  ?? 0),
    'productK'  => to_float($p['productK']  ?? 0),
    'productS'  => to_float($p['productS']  ?? 0),
    'productUD' => to_float($p['productUD'] ?? 0),
    'productCa' => to_float($p['productCa'] ?? 0),
    'productMg' => to_float($p['productMg'] ?? 0),
    'productZn' => to_float($p['productZn'] ?? 0),
    'productFe' => to_float($p['productFe'] ?? 0),
    'productMn' => to_float($p['productMn'] ?? 0),
    'productCu' => to_float($p['productCu'] ?? 0),
    'productB'  => to_float($p['productB']  ?? 0),
    'productMo' => to_float($p['productMo'] ?? 0),

    // optional extra fields (safe)
    'productQuantity'    => to_float($p['productQuantity'] ?? 0),
    'altProductQuantity' => to_float($p['altProductQuantity'] ?? 0),
    'productWeight'      => to_float($p['productWeight'] ?? 0),
    'productAcres'       => to_float($p['productAcres'] ?? 0),
    'productDerived1'    => norm_str($p['productDerived1'] ?? ''),
    'productDerived2'    => norm_str($p['productDerived2'] ?? ''),
    'productOverview'    => norm_str($p['productOverview'] ?? ''),
  ];
}

fclose($handle);

echo json_encode($products, JSON_UNESCAPED_UNICODE);
