// Global product data
let productData = [];

// Load products from server
async function loadProductData() {
  const response = await fetch('load_products.php');
  productData = await response.json();
  document.getElementById('addBtn').disabled = false;
  document.getElementById('clearBtn').disabled = false;
  addProductRow();
}

// Add a new product row
function addProductRow() {
  const container = document.getElementById('productContainer');
  const row = document.createElement('div');
  row.className = 'product-row';

  const select = document.createElement('select');
  const defaultOption = document.createElement('option');
  defaultOption.value = '';
  defaultOption.textContent = 'Choose product...';
  defaultOption.disabled = true;
  defaultOption.selected = true;
  select.appendChild(defaultOption);

  productData.forEach(product => {
    if (product.productType.toLowerCase() === 'dry') {
      const option = document.createElement('option');
      option.value = product.productId;
      option.textContent = `${product.productAlias} — ${product.productName}`;
      select.appendChild(option);
    }
  });
  select.onchange = updateTotals;

  const rateInput = document.createElement('input');
  rateInput.type = 'number';
  rateInput.placeholder = 'lbs/acre';
  rateInput.oninput = updateTotals;

  const priceInput = document.createElement('input');
  priceInput.type = 'number';
  priceInput.placeholder = '$/ton';
  priceInput.className = 'price-input';
  priceInput.oninput = updateTotals;

  const removeBtn = document.createElement('span');
  removeBtn.textContent = 'X';
  removeBtn.className = 'remove-btn';
  removeBtn.onclick = () => {
    row.remove();
    updateTotals();
  };

  row.appendChild(select);
  row.appendChild(rateInput);
  row.appendChild(priceInput);
  row.appendChild(removeBtn);
  container.appendChild(row);
}

// Clear all rows
function clearAllRows() {
  document.getElementById('productContainer').innerHTML = '';
  updateTotals();
}

// Update all calculations and UI
function updateTotals() {
  let totals = {
    N: 0, P: 0, K: 0, Ca: 0, S: 0, Mg: 0, Zn: 0,
    Fe: 0, Mn: 0, Cu: 0, B: 0
  };
  let totalLbs = 0;
  let weightedDensitySum = 0;
  let totalCost = 0;
  let totalWeight = 0;

  const blendFee = parseFloat(document.getElementById('blendFee').value) || 0;
  const rows = document.querySelectorAll('.product-row');

  rows.forEach(row => {
    const select = row.querySelector('select');
    const input = row.querySelector('input');
    const priceInput = row.querySelector('.price-input');

    const product = productData.find(p => p.productId === select.value);
    const rate = parseFloat(input.value) || 0;
    const pricePerUnit = parseFloat(priceInput?.value) || 0;

    if (product) {
      Object.keys(totals).forEach(key => {
        totals[key] += (product[`product${key}`] || 0) / 100 * rate;
      });
      totalLbs += rate;
      weightedDensitySum += rate * (product.productDensity || 0);
      const productWeightTons = rate / 2000;
      totalCost += productWeightTons * pricePerUnit;
      totalWeight += productWeightTons;
    }
  });

  const avgDensity = totalLbs > 0 ? (weightedDensitySum / totalLbs) : 0;
  const priceTotal = totalCost + (totalWeight * blendFee);
  const pricePerTon = totalWeight ? (priceTotal / totalWeight) : 0;

  document.getElementById('totalN').textContent = totals.N.toFixed(2);
  document.getElementById('totalP').textContent = totals.P.toFixed(2);
  document.getElementById('totalK').textContent = totals.K.toFixed(2);
  document.getElementById('totalCa').textContent = totals.Ca.toFixed(2);
  document.getElementById('totalS').textContent = totals.S.toFixed(2);
  document.getElementById('totalMg').textContent = totals.Mg.toFixed(2);
  document.getElementById('totalZn').textContent = totals.Zn.toFixed(2);
  document.getElementById('totalFe').textContent = totals.Fe.toFixed(2);
  document.getElementById('totalMn').textContent = totals.Mn.toFixed(2);
  document.getElementById('totalCu').textContent = totals.Cu.toFixed(2);
  document.getElementById('totalB').textContent = totals.B.toFixed(2);

  document.getElementById('priceTotal').textContent = priceTotal.toFixed(2);
  document.getElementById('pricePerTon').textContent = pricePerTon.toFixed(2);
  document.getElementById('totalProduct').textContent = totalLbs.toFixed(2);
  document.getElementById('blendDensity').textContent = avgDensity.toFixed(2);
  document.getElementById('blendWeight').textContent = totalWeight.toFixed(2);

  document.getElementById('uN').textContent = totals.N.toFixed(2);
  document.getElementById('uP').textContent = totals.P.toFixed(2);
  document.getElementById('uK').textContent = totals.K.toFixed(2);
  document.getElementById('uCa').textContent = totals.Ca.toFixed(2);
  document.getElementById('uS').textContent = totals.S.toFixed(2);

  document.getElementById('aN').textContent = totalLbs ? (totals.N / totalLbs * 100).toFixed(1) : '0';
  document.getElementById('aP').textContent = totalLbs ? (totals.P / totalLbs * 100).toFixed(1) : '0';
  document.getElementById('aK').textContent = totalLbs ? (totals.K / totalLbs * 100).toFixed(1) : '0';
  document.getElementById('aCa').textContent = totalLbs ? (totals.Ca / totalLbs * 100).toFixed(1) : '0';
  document.getElementById('aS').textContent = totalLbs ? (totals.S / totalLbs * 100).toFixed(1) : '0';
}

// Event listeners
window.onload = () => {
  loadProductData();
  document.getElementById('addBtn').onclick = addProductRow;
  document.getElementById('clearBtn').onclick = clearAllRows;
  document.getElementById('blendFee').oninput = updateTotals;
  document.getElementById('showPricesToggle').onchange = () => {
    document.body.classList.toggle('show-prices');
  };
  document.getElementById('printBtn').onclick = () => {
  generatePrintLayout();

  // Temporarily hide the calculator UI, show print layout
  const mainUI = document.body.children;
  for (let i = 0; i < mainUI.length; i++) {
    if (mainUI[i].id !== 'printOnlyOutput') {
      mainUI[i].setAttribute('data-print-hide', 'true');
      mainUI[i].style.display = 'none';
    }
  }

  document.getElementById('printOnlyOutput').style.display = 'block';

  // Give time for DOM to update, then print
  setTimeout(() => {
    window.print();

    // After printing, restore original UI
    for (let i = 0; i < mainUI.length; i++) {
      if (mainUI[i].getAttribute('data-print-hide') === 'true') {
        mainUI[i].style.display = '';
        mainUI[i].removeAttribute('data-print-hide');
      }
    }
    document.getElementById('printOnlyOutput').style.display = 'none';
  }, 200);
};

};

};
