<?php
// === db.php (mysqli-only, with auth, admin, passwords, settings) ===
$config = require __DIR__ . '/config.php';
if (file_exists(__DIR__ . '/config.local.php')) {
  $config = array_merge($config, require __DIR__ . '/config.local.php');
}

ini_set('display_errors', $config['app_debug'] ? '1' : '0');
error_reporting($config['app_debug'] ? E_ALL : 0);

session_name($config['session_name'] ?? 'fb_min_sess');
session_start();

// CSRF
if (empty($_SESSION['csrf'])) $_SESSION['csrf'] = bin2hex(random_bytes(16));
function require_csrf() {
  $m = $_SERVER['REQUEST_METHOD'] ?? 'GET';
  if (in_array($m, ['POST','PUT','PATCH','DELETE'])) {
    $hdr = $_SERVER['HTTP_X_CSRF_TOKEN'] ?? '';
    if (!$hdr || !hash_equals($_SESSION['csrf'], $hdr)) {
      http_response_code(419);
      header('Content-Type: application/json');
      echo json_encode(['error'=>'csrf_failed']); exit;
    }
  }
}

// DSN parse
$host='localhost'; $port=3306; $dbname='featureboard';
if (preg_match('/host=([^;]+)/',$config['db_dsn'],$m)) $host=$m[1];
if (preg_match('/port=(\d+)/',$config['db_dsn'],$m)) $port=(int)$m[1];
if (preg_match('/dbname=([^;]+)/',$config['db_dsn'],$m)) $dbname=$m[1];

$mysqli = mysqli_init();
$mysqli->options(MYSQLI_OPT_INT_AND_FLOAT_NATIVE, 1);
if (!$mysqli->real_connect($host, $config['db_user'], $config['db_pass'], $dbname, $port)) {
  http_response_code(500);
  echo "DB connect failed (mysqli): " . htmlspecialchars($mysqli->connect_error); exit;
}
$mysqli->set_charset('utf8mb4');

class DB {
  public $db;
  function __construct($db){ $this->db=$db; }
  function queryAll($sql,$params=[]){
    $stmt=$this->prep($sql,$params); $res=$stmt->get_result();
    $rows=[]; if ($res) { while($r=$res->fetch_assoc()) $rows[]=$r; }
    $stmt->close(); return $rows;
  }
  function queryOne($sql,$params=[]){
    $rows=$this->queryAll($sql,$params); return $rows ? $rows[0] : null;
  }
  function exec($sql,$params=[]){
    $stmt=$this->prep($sql,$params); $stmt->close(); return true;
  }
  function lastId(){ return $this->db->insert_id; }
  private function prep($sql,$params){
    $stmt=$this->db->prepare($sql);
    if(!$stmt) throw new Exception($this->db->error);
    if ($params){
      $types=''; $values=[];
      foreach($params as $p){
        if (is_int($p)) { $types.='i'; }
        elseif (is_float($p)) { $types.='d'; }
        else { $types.='s'; }
        $values[]=$p;
      }
      $stmt->bind_param($types, ...$values);
    }
    if(!$stmt->execute()) throw new Exception($stmt->error);
    return $stmt;
  }
}
$db = new DB($mysqli);

// Settings
function get_setting(DB $db, $key, $default=null) {
  $row = $db->queryOne("SELECT val FROM settings WHERE `key`=?", [$key]);
  return $row ? $row['val'] : $default;
}
function set_setting(DB $db, $key, $val) {
  $row = $db->queryOne("SELECT 1 FROM settings WHERE `key`=?", [$key]);
  if ($row) $db->exec("UPDATE settings SET val=? WHERE `key`=?", [$val, $key]);
  else $db->exec("INSERT INTO settings (`key`,val) VALUES (?,?)", [$key,$val]);
}

// Auth helpers
function normalize_email($email){ return strtolower(trim($email)); }
function current_user_id(DB $db) { return $_SESSION['user_id'] ?? 0; }
function current_user(DB $db) {
  if (!current_user_id($db)) return null;
  return $db->queryOne("SELECT id,email,name,role FROM users WHERE id=?", [$_SESSION['user_id']]);
}
function require_user(DB $db) {
  if (!current_user_id($db)) {
    http_response_code(401);
    header('Content-Type: application/json'); echo json_encode(['error'=>'auth_required']); exit;
  }
}
function require_admin(DB $db) {
  $u = current_user($db);
  if (!$u || $u['role']!=='admin') {
    http_response_code(403);
    header('Content-Type: application/json'); echo json_encode(['error'=>'admin_required']); exit;
  }
}
function auth_with_email(DB $db, $email) {
  $email = normalize_email($email);
  if (!filter_var($email, FILTER_VALIDATE_EMAIL)) return [false, 'invalid_email'];
  $allowed = get_setting($db, 'allowed_domain', '');
  if ($allowed) {
    $domain = substr(strrchr($email,'@'),1);
    if (strtolower($domain) !== strtolower($allowed)) return [false, 'domain_not_allowed'];
  }
  $row = $db->queryOne("SELECT id FROM users WHERE email=?", [$email]);
  if (!$row) {
    $db->exec("INSERT INTO users (email,name,role) VALUES (?,?,?)", [$email, $email, 'user']);
    $uid = (int)$db->lastId();
  } else {
    $uid = (int)$row['id'];
  }
  $_SESSION['user_id'] = $uid;
  return [true, 'ok'];
}

// Admin password auth (name or email + password)
function admin_login(DB $db, $identifier, $password){
  $identifier = trim($identifier);
  $u = $db->queryOne("SELECT id,email,name,role,password_hash FROM users WHERE email=? OR name=? LIMIT 1", [$identifier,$identifier]);
  if (!$u || $u['role']!=='admin') return [false,'not_admin'];
  if (!$u['password_hash']) return [false,'no_password_set'];
  if (!password_verify($password, $u['password_hash'])) return [false,'bad_password'];
  $_SESSION['user_id'] = (int)$u['id'];
  return [true,'ok'];
}
