<?php
require __DIR__ . '/../db.php';
header('Content-Type: application/json');

$action = $_GET['action'] ?? '';
if (!in_array($action, ['features_list','csrf','settings_get','categories_list','duplicate_suggest','duplicate_suggest_lev','admin_me'])) {
  require_csrf();
}

function read_json() {
  $in = file_get_contents('php://input');
  $data = json_decode($in, true);
  return is_array($data) ? $data : [];
}
function out($data, $code=200){ http_response_code($code); echo json_encode($data); exit; }

switch ($action) {
  case 'csrf':
    out(['csrf'=>$_SESSION['csrf']]);

  case 'admin_login':
    $d = read_json();

    // If default admin has no password, set it on first login attempt:
    if (($d['identifier'] ?? '') === 'FeatureAdmin') {
      $row = $db->queryOne("SELECT id,password_hash FROM users WHERE name='FeatureAdmin' AND role='admin' LIMIT 1", []);
      if ($row && !$row['password_hash']) {
        $hash = password_hash('!1Feature3#', PASSWORD_BCRYPT);
        $db->exec("UPDATE users SET password_hash=? WHERE id=?", [$hash, $row['id']]);
      }
    }

    list($ok,$msg) = admin_login($db, $d['identifier'] ?? '', $d['password'] ?? '');
    if (!$ok) out(['error'=>$msg], 401);
    out(['ok'=>true]);

  case 'admin_me':
    $u = current_user($db);
    out(['user'=>$u]);

  case 'settings_get':
    out(['allowed_domain'=> get_setting($db,'allowed_domain','')]);

  case 'settings_set_domain':
    require_admin($db);
    $data = read_json();
    $dom = strtolower(trim($data['allowed_domain'] ?? ''));
    if (!$dom) out(['error'=>'invalid_domain'],422);
    set_setting($db,'allowed_domain',$dom);
    out(['ok'=>true]);

  case 'users_set_role':
    require_admin($db);
    $data = read_json();
    $email = strtolower(trim($data['email'] ?? ''));
    $role  = $data['role'] ?? 'user';
    if (!in_array($role,['user','admin'])) out(['error'=>'invalid_role'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $db->exec("UPDATE users SET role=? WHERE id=?",[$role, $u['id']]);
    out(['ok'=>true]);

  case 'users_set_password':
    require_admin($db);
    $data = read_json();
    $email = strtolower(trim($data['email'] ?? ''));
    $pass  = $data['password'] ?? '';
    if (!$email || strlen($pass)<6) out(['error'=>'invalid'],422);
    $u = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if (!$u) out(['error'=>'user_not_found'],404);
    $hash = password_hash($pass, PASSWORD_BCRYPT);
    $db->exec("UPDATE users SET password_hash=? WHERE id=?",[$hash,$u['id']]);
    out(['ok'=>true]);

  case 'users_create_admin':
    require_admin($db);
    $d = read_json();
    $email = strtolower(trim($d['email'] ?? ''));
    $name  = trim($d['name'] ?? $email);
    $pass  = $d['password'] ?? '';
    if (!filter_var($email,FILTER_VALIDATE_EMAIL) || strlen($pass)<6) out(['error'=>'invalid'],422);
    $row = $db->queryOne("SELECT id FROM users WHERE email=?",[$email]);
    if ($row) out(['error'=>'exists'],409);
    $hash = password_hash($pass, PASSWORD_BCRYPT);
    $db->exec("INSERT INTO users (email,name,role,password_hash) VALUES (?,?,?,?)",[$email,$name,'admin',$hash]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'categories_list':
    $rows = $db->queryAll("SELECT id,name,sort_order FROM categories ORDER BY sort_order ASC, name ASC");
    out($rows);

  case 'categories_create':
    require_admin($db);
    $data = read_json();
    $name = trim($data['name'] ?? '');
    $sort = (int)($data['sort_order'] ?? 100);
    if ($name==='') out(['error'=>'name_required'],422);
    $db->exec("INSERT INTO categories (name,sort_order) VALUES (?,?)",[$name,$sort]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'categories_delete':
    require_admin($db);
    $data = read_json();
    $id = (int)($data['id'] ?? 0);
    if (!$id) out(['error'=>'id_required'],422);
    $db->exec("DELETE FROM categories WHERE id=?",[$id]);
    out(['ok'=>true]);

  case 'features_list':
    $sort = $_GET['sort'] ?? 'top';
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : null;
    $status = isset($_GET['status']) ? (int)$_GET['status'] : null;

    $where=[]; $params=[];
    if ($category_id){ $where[]="f.category_id = ?"; $params[]=$category_id; }
    if ($status){ $where[]="f.status_id = ?"; $params[]=$status; }
    $w = $where ? ("WHERE ".implode(" AND ", $where)) : "";

    if ($sort==='new') $order="ORDER BY f.created_at DESC";
    elseif ($sort==='trending') $order="ORDER BY recent_votes DESC, total_votes DESC, f.created_at DESC";
    else $order="ORDER BY total_votes DESC, f.created_at DESC";

    $sql = "
      SELECT f.*, COALESCE(tv.total_votes,0) AS total_votes, COALESCE(rv.recent_votes,0) AS recent_votes,
             c.name AS category_name
      FROM features f
      LEFT JOIN (
        SELECT feature_id, COUNT(*) total_votes FROM feature_votes GROUP BY feature_id
      ) tv ON tv.feature_id=f.id
      LEFT JOIN (
        SELECT feature_id, COUNT(*) recent_votes FROM feature_votes
        WHERE created_at >= (NOW() - INTERVAL 30 DAY)
        GROUP BY feature_id
      ) rv ON rv.feature_id=f.id
      LEFT JOIN categories c ON c.id = f.category_id
      $w
      $order
      LIMIT 200";
    $rows = $db->queryAll($sql,$params);
    out($rows);

  case 'features_create':
    require_user($db);
    $data = read_json();
    $title = trim($data['title'] ?? '');
    $desc  = trim($data['description'] ?? '');
    $catId = isset($data['category_id']) ? (int)$data['category_id'] : null;
    if ($title==='') out(['error'=>'title_required'],422);

    $u = current_user($db);
    $allowed = get_setting($db,'allowed_domain','');
    if ($allowed) {
      $domain = substr(strrchr($u['email'],'@'),1);
      if (strtolower($domain)!==strtolower($allowed)) out(['error'=>'domain_not_allowed'],403);
    }

    $db->exec("INSERT INTO features (title,description,author_id,category_id) VALUES (?,?,?,?)",
              [$title,$desc,$u['id'],$catId ?: NULL]);
    out(['ok'=>true,'id'=>$db->lastId()]);

  case 'features_mark_duplicate':
    require_admin($db);
    $d = read_json();
    $fid = (int)($d['feature_id'] ?? 0);
    $dup = (int)($d['duplicate_of_id'] ?? 0);
    if (!$fid || !$dup || $fid===$dup) out(['error'=>'invalid_ids'],422);
    $has = $db->queryOne("SELECT id FROM features WHERE id=?",[$dup]);
    if (!$has) out(['error'=>'dup_not_found'],404);
    $db->exec("UPDATE features SET duplicate_of=? WHERE id=?",[$dup,$fid]);
    out(['ok'=>true]);

  case 'vote_toggle':
    require_user($db);
    $data = read_json();
    $fid = (int)($data['feature_id'] ?? 0);
    if (!$fid) out(['error'=>'feature_id_required'],422);
    $uid = current_user_id($db);
    $exists = $db->queryOne("SELECT 1 FROM feature_votes WHERE feature_id=? AND user_id=?",[$fid,$uid]);
    if ($exists){
      $db->exec("DELETE FROM feature_votes WHERE feature_id=? AND user_id=?",[$fid,$uid]);
      out(['ok'=>true,'action'=>'unvoted']);
    } else {
      $db->exec("INSERT INTO feature_votes (feature_id,user_id) VALUES (?,?)",[$fid,$uid]);
      out(['ok'=>true,'action'=>'voted']);
    }

  case 'comments_create':
    require_user($db);
    $data = read_json();
    $fid = (int)($data['feature_id'] ?? 0);
    $body = trim($data['body'] ?? '');
    if (!$fid || $body==='') out(['error'=>'invalid'],422);
    $db->exec("INSERT INTO feature_comments (feature_id,user_id,body) VALUES (?,?,?)",[ $fid, current_user_id($db), $body ]);
    out(['ok'=>true]);

  case 'feature_set_status':
    require_admin($db);
    $data = read_json();
    $fid = (int)($data['feature_id'] ?? 0);
    $sid = (int)($data['status_id'] ?? 0);
    if (!$fid || !$sid) out(['error'=>'invalid'],422);
    $db->exec("UPDATE features SET status_id=? WHERE id=?",[$sid,$fid]);
    out(['ok'=>true]);

  case 'duplicate_suggest':
    $q = trim($_GET['q'] ?? '');
    if ($q==='') out([]);
    $rows = $db->queryAll("
      SELECT id,title,COALESCE(tv.total_votes,0) AS votes
      FROM features f
      LEFT JOIN (SELECT feature_id, COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
      ON tv.feature_id=f.id
      WHERE title LIKE ? OR SOUNDEX(title)=SOUNDEX(?)
      ORDER BY votes DESC, created_at DESC
      LIMIT 10
    ", ['%'+$q+'%', $q]);
    // Fix PHP string concatenation issue: redo with proper params
    $rows = $db->queryAll("
      SELECT id,title,COALESCE(tv.total_votes,0) AS votes
      FROM features f
      LEFT JOIN (SELECT feature_id, COUNT(*) total_votes FROM feature_votes GROUP BY feature_id) tv
      ON tv.feature_id=f.id
      WHERE title LIKE ? OR SOUNDEX(title)=SOUNDEX(?)
      ORDER BY votes DESC, created_at DESC
      LIMIT 10
    ", ['%'.$q.'%', $q]);
    out($rows);

  case 'duplicate_suggest_lev':
    $q = trim($_GET['q'] ?? '');
    if ($q==='') out([]);
    $all = $db->queryAll("SELECT id,title FROM features ORDER BY id DESC LIMIT 500");
    $res = [];
    foreach ($all as $r) {
      $dist = levenshtein(mb_strtolower($q,'UTF-8'), mb_strtolower($r['title'],'UTF-8'));
      $res[] = ['id'=>$r['id'], 'title'=>$r['title'], 'distance'=>$dist];
    }
    usort($res, function($a,$b){ return $a['distance'] <=> $b['distance']; });
    $res = array_slice($res, 0, 10);
    out($res);

  default:
    out(['error'=>'unknown_action'],400);
}
